import { AntDesign as Icon } from '@expo/vector-icons';
import React, { useState } from 'react';
import {
    Image,
    SafeAreaView,
    StyleSheet,
    Text,
    TouchableOpacity,
    View,
} from 'react-native';
import Modal from 'react-native-modal';
import Colors from '../../constants/colors';
import { ArrayComponent } from '../common';

interface FormikDocumentPickerProps {
  field: { name: string; value?: any[] };
  setFieldValue: (field: string, value: any) => void;
  value?: any[];
  backgroundColor?: string;
  textColor?: string;
  disabled?: boolean;
}

type DocumentType = 'image' | 'video' | 'docs';

const FormikDocumentPicker: React.FC<FormikDocumentPickerProps> = ({
  field,
  setFieldValue,
  value = [],
  backgroundColor = '#b69b30',
  textColor = '#fff',
  disabled = true,
}) => {
  const [isAttachModalVisible, setAttachModalVisible] = useState(false);
  const [isPickerModalVisible, setPickerModalVisible] = useState(false);
  const [type, setType] = useState<DocumentType>('image');

  const getImageByType = (docType: DocumentType) => {
    switch (docType) {
      case 'image':
        return require('@/assets/images/camera.png');
      case 'video':
        return require('@/assets/images/video.png');
      case 'docs':
        return require('@/assets/images/upload_file.png');
      default:
        return require('@/assets/images/camera.png');
    }
  };

  const openPicker = (docType: DocumentType) => {
    setType(docType);
    setPickerModalVisible(true);
  };

  return (
    <>
      <SafeAreaView>
        {/* Attach Modal */}
        <Modal
          isVisible={isAttachModalVisible}
          onBackButtonPress={() => setAttachModalVisible(false)}
          onBackdropPress={() => setAttachModalVisible(false)}
        >
          <View style={styles.modalContent}>
            <Icon
              name="close-circle"
              size={24}
              color={Colors.colorPlaceholder}
              style={styles.closeIcon}
              onPress={() => setAttachModalVisible(false)}
            />

            <View style={styles.attachOptions}>
              <TouchableOpacity
                style={styles.attachButton}
                onPress={() => openPicker('image')}
              >
                <Text style={styles.attachButtonText}>Image</Text>
                <Image
                  source={require('@/assets/images/camera.png')}
                  style={styles.logo}
                />
              </TouchableOpacity>

              <TouchableOpacity
                style={styles.attachButton}
                onPress={() => openPicker('docs')}
              >
                <Text style={styles.attachButtonText}>File</Text>
                <Image
                  source={require('@/assets/images/upload_file.png')}
                  style={styles.logo}
                />
              </TouchableOpacity>
            </View>
          </View>
        </Modal>

        {/* Document Picker Modal */}
        <Modal
          isVisible={isPickerModalVisible}
          onBackButtonPress={() => setPickerModalVisible(false)}
          onBackdropPress={() => setPickerModalVisible(false)}
        >
          <View style={styles.modalContent}>
            <Icon
              name="close-circle"
              size={24}
              color={Colors.colorPlaceholder}
              style={styles.closeIcon}
              onPress={() => setPickerModalVisible(false)}
            />

            <ArrayComponent
              value={value}
              type={type}
              updateFormikData={(data) => {
                setFieldValue(field.name, data);
                setPickerModalVisible(false);
                setAttachModalVisible(false);
              }}
            />
          </View>
        </Modal>
      </SafeAreaView>

      {/* Attach Button */}
      <TouchableOpacity
        style={[
          styles.attachContainer,
          {
            backgroundColor: value.length > 0 ? backgroundColor : '#ECECEC',
          },
        ]}
        onPress={() => {
          disabled && setAttachModalVisible(true);
        }}
      >
        <Text
          style={[
            styles.attachText,
            { color: value.length > 0 ? textColor : Colors.textColor },
          ]}
        >
          Attach
        </Text>
        <Image source={getImageByType(type)} style={styles.logo} />
      </TouchableOpacity>
    </>
  );
};

const styles = StyleSheet.create({
  modalContent: {
    backgroundColor: 'white',
    padding: 16,
    borderRadius: 8,
    alignItems: 'center',
  },
  closeIcon: {
    alignSelf: 'flex-end',
    marginBottom: 12,
  },
  attachOptions: {
    flexDirection: 'row',
    justifyContent: 'space-around',
    width: '100%',
  },
  attachButton: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: Colors.colorTextInput,
    paddingHorizontal: 12,
    paddingVertical: 8,
    borderRadius: 8,
  },
  attachButtonText: {
    color: '#fff',
    fontSize: 14,
    fontWeight: '600',
    marginRight: 6,
  },
  attachContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    borderRadius: 20,
    borderWidth: 1,
    borderColor: '#ECECEC',
    padding: 10,
    height: 40,
    width: '30%',
    marginLeft: 8,
  },
  attachText: {
    fontSize: 12,
    textAlign: 'center',
  },
  logo: {
    width: 24,
    height: 24,
    resizeMode: 'contain',
    marginLeft: 4,
    tintColor: '#fff',
  },
});

export { FormikDocumentPicker };
