// reducers/permitReducer.ts
import { Toast } from 'native-base';
import { REHYDRATE } from 'redux-persist';
import {
    GET_PERMIT_FORM_DATA_FAILED,
    GET_PERMIT_FORM_DATA_REQUEST,
    GET_PERMIT_FORM_DATA_SUCCESS,
    SUBMIT_PERMIT_FAILED,
    SUBMIT_PERMIT_REQUEST,
    SUBMIT_PERMIT_SUCCESS,
} from '../actions/types';

// =======================
// Types
// =======================
export interface PermitFormDataState {
  field_job_location: any[];
  webform: any[];
  field_invited_contractor: any[];
  loading: boolean;
  error: Record<string, any>;
}

export interface PermitSubmitDataState {
  data: Record<string, any>;
  loading: boolean;
  error: Record<string, any>;
}

// Action Types
type PermitFormAction =
  | { type: typeof REHYDRATE; payload?: { permitFormData?: PermitFormDataState } }
  | { type: typeof GET_PERMIT_FORM_DATA_REQUEST }
  | { type: typeof GET_PERMIT_FORM_DATA_SUCCESS; payload: PermitFormDataState }
  | { type: typeof GET_PERMIT_FORM_DATA_FAILED; payload: Record<string, any> };

type PermitSubmitAction =
  | { type: typeof SUBMIT_PERMIT_REQUEST }
  | { type: typeof SUBMIT_PERMIT_SUCCESS; payload: Record<string, any> }
  | { type: typeof SUBMIT_PERMIT_FAILED; payload: Record<string, any> };

// =======================
// Initial States
// =======================
const permitFormInitialState: PermitFormDataState = {
  field_job_location: [],
  webform: [],
  field_invited_contractor: [],
  loading: false,
  error: {},
};

const permitSubmitInitialState: PermitSubmitDataState = {
  data: {},
  loading: false,
  error: {},
};

// =======================
// Reducers
// =======================
export const permitFormData = (
  state: PermitFormDataState = permitFormInitialState,
  action: PermitFormAction
): PermitFormDataState => {
  switch (action.type) {
    case REHYDRATE:
      return action.payload?.permitFormData
        ? {
            ...state,
            field_job_location: [...(action.payload.permitFormData.field_job_location || [])],
            webform: [...(action.payload.permitFormData.webform || [])],
            field_invited_contractor: [...(action.payload.permitFormData.field_invited_contractor || [])],
            loading: false,
            error: {},
          }
        : state;

    case GET_PERMIT_FORM_DATA_REQUEST:
      return { ...state, loading: true, error: {} };

    case GET_PERMIT_FORM_DATA_SUCCESS:
      return {
        ...state,
        field_job_location: [...(action.payload.field_job_location || [])],
        webform: [...(action.payload.webform || [])],
        field_invited_contractor: [...(action.payload.field_invited_contractor || [])],
        loading: false,
        error: {},
      };

    case GET_PERMIT_FORM_DATA_FAILED:
      return { ...state, loading: false, error: action.payload || {} };

    default:
      return state;
  }
};

export const permitSubmitData = (
  state: PermitSubmitDataState = permitSubmitInitialState,
  action: PermitSubmitAction
): PermitSubmitDataState => {
  switch (action.type) {
    case SUBMIT_PERMIT_REQUEST:
      return { ...state, loading: true, error: {} };

    case SUBMIT_PERMIT_SUCCESS:
      Toast.show({
        title: 'Permit Submitted',
        placement: 'top',
        description: 'Permit submitted successfully!',
        duration: 4000,
        variant: 'solid',
        backgroundColor: 'green.500',
      });
      return { ...state, data: action.payload, loading: false, error: {} };

    case SUBMIT_PERMIT_FAILED:
      Toast.show({
        title: 'Submission Failed',
        placement: 'top',
        description: 'Something went wrong while submitting permit.',
        duration: 4000,
        variant: 'solid',
        backgroundColor: 'red.500',
      });
      return { ...state, loading: false, error: action.payload || {} };

    default:
      return state;
  }
};
