import { Entypo as EntypoIcon } from '@expo/vector-icons';
import { Box, HStack, Icon, Modal, Pressable, ScrollView, Text, VStack } from 'native-base';
import React, { useEffect, useState } from 'react';
import { Alert, Image } from 'react-native';
import { useDispatch, useSelector } from 'react-redux';
import Colors from '../../constants/colors';
import { getNotifications } from '../../store/actions/notificationActions';
import type { AppDispatch } from '../../store/store'; // Add this import - adjust path to your store
import { setAuthentication } from '../../utils/authentication';

interface MenuButtonProps {
  navigation: any;
  logout: () => void;
  online: boolean;
  syncData: () => void;
  access_token?: string;
}

export const MenuButton: React.FC<MenuButtonProps> = ({ navigation, logout, online, syncData, access_token }) => {
  const [visible, setVisible] = useState(false);
  const state = useSelector((state: any) => state.notificationData);
  const notifications = state.notifications || [];
  const unread_count = state.unread_count || 0;
  const dispatch = useDispatch<AppDispatch>(); // Add AppDispatch type here
  const offline = useSelector((state: any) => state.offline);

  if (access_token) {
    setAuthentication(access_token);
  }

  useEffect(() => {
    if (access_token && online) {
      dispatch(getNotifications());
    }
  }, [online]);

  const handleLogout = () => {
    Alert.alert(
      'Logout',
      'Are you sure you want to logout?',
      [
        { text: 'Cancel', style: 'cancel' },
        { text: 'OK', onPress: logout }
      ],
      { cancelable: false }
    );
    setVisible(false);
  };

  return (
    <HStack alignItems="center">
      {/* Notification Icon */}
      <Pressable
        onPress={() =>
          notifications.length > 0
            ? navigation.navigate('Notifications', { online: offline.online })
            : alert('You have no notifications')
        }
      >
        <Image
          source={require('../../assets/images/notification.png')}
          style={{ width: 30, height: 35, margin: 10 }}
        />
        {unread_count > 0 && (
          <Box
            position="absolute"
            left={25}
            bottom={27}
            width={25}
            height={25}
            borderRadius={12.5}
            bg="#8A1538"
            alignItems="center"
            justifyContent="center"
          >
            <Text color="white" fontSize={10} fontWeight="bold">
              {unread_count}
            </Text>
          </Box>
        )}
      </Pressable>

      {/* Menu Icon */}
      <Pressable onPress={() => setVisible(true)}>
        <Icon as={EntypoIcon} name="dots-three-vertical" size={8} color={Colors.buttonColor} mr={2} />
      </Pressable>

      {/* Modal */}
      <Modal isOpen={visible} onClose={() => setVisible(false)} safeAreaTop>
        <Modal.Content maxWidth="350px" alignSelf="flex-end">
          <Modal.Body p={0}>
            <ScrollView>
              <VStack>
                {online && (
                  <>
                    <Pressable
                      onPress={() => {
                        syncData();
                        setVisible(false);
                      }}
                    >
                      <HStack alignItems="center" justifyContent="space-between" p={3}>
                        <Text fontWeight="600">Update Forms</Text>
                        <Icon as={EntypoIcon} name="cycle" size={6} color={Colors.buttonColor} />
                      </HStack>
                    </Pressable>
                    <Box h={1} bg="#E6E6E6" />
                  </>
                )}

                <Pressable
                  onPress={() => {
                    navigation.navigate('OfflineLog');
                    setVisible(false);
                  }}
                >
                  <HStack alignItems="center" justifyContent="space-between" p={3}>
                    <Text fontWeight="600">Pending Jobs</Text>
                    <Icon as={EntypoIcon} name="export" size={6} color={Colors.buttonColor} />
                  </HStack>
                </Pressable>
                <Box h={1} bg="#E6E6E6" />

                <Pressable
                  onPress={() => {
                    navigation.navigate('FailedJobs');
                    setVisible(false);
                  }}
                >
                  <HStack alignItems="center" justifyContent="space-between" p={3}>
                    <Text fontWeight="600">Retry Jobs</Text>
                    <Icon as={EntypoIcon} name="ccw" size={6} color={Colors.buttonColor} />
                  </HStack>
                </Pressable>
                <Box h={1} bg="#E6E6E6" />

                <Pressable
                  onPress={() => {
                    navigation.navigate('History');
                    setVisible(false);
                  }}
                >
                  <HStack alignItems="center" justifyContent="space-between" p={3}>
                    <Text fontWeight="600">History</Text>
                    <Icon as={EntypoIcon} name="back-in-time" size={6} color={Colors.buttonColor} />
                  </HStack>
                </Pressable>
                <Box h={1} bg="#E6E6E6" />

                <Pressable
                  onPress={() => {
                    navigation.navigate('ResetPasswordScreen');
                    setVisible(false);
                  }}
                >
                  <HStack alignItems="center" justifyContent="space-between" p={3}>
                    <Text fontWeight="600">Reset Password</Text>
                    <Icon as={EntypoIcon} name="retweet" size={6} color={Colors.buttonColor} />
                  </HStack>
                </Pressable>
                <Box h={1} bg="#E6E6E6" />

                {online && (
                  <>
                    <Pressable onPress={handleLogout}>
                      <HStack alignItems="center" justifyContent="space-between" p={3}>
                        <Text fontWeight="600">Logout</Text>
                        <Icon as={EntypoIcon} name="log-out" size={6} color={Colors.buttonColor} />
                      </HStack>
                    </Pressable>
                    <Box h={1} bg="#E6E6E6" />
                  </>
                )}

                <Pressable
                  onPress={() => {
                    navigation.navigate('ContactUsPage');
                    setVisible(false);
                  }}
                >
                  <HStack alignItems="center" justifyContent="space-between" p={3}>
                    <Text fontWeight="600">Contact Us</Text>
                    <Icon as={EntypoIcon} name="phone" size={6} color={Colors.buttonColor} />
                  </HStack>
                </Pressable>
              </VStack>
            </ScrollView>
          </Modal.Body>
        </Modal.Content>
      </Modal>
    </HStack>
  );
};