// reducers/inspectionReducer.ts
import { Toast } from 'native-base';
import { REHYDRATE } from 'redux-persist';
import {
    GET_INSPECTION_FORM_DATA_FAILED,
    GET_INSPECTION_FORM_DATA_REQUEST,
    GET_INSPECTION_FORM_DATA_SUCCESS,
    SUBMIT_INSPECTION_FAILED,
    SUBMIT_INSPECTION_REQUEST,
    SUBMIT_INSPECTION_SUCCESS,
} from '../actions/types';

// =======================
// Types
// =======================
export interface InspectionFormDataState {
  data: Record<string, any>;
  loading: boolean;
  error: Record<string, any>;
}

// =======================
// Action Types
// =======================
type InspectionFormAction =
  | { type: typeof REHYDRATE; payload?: { inspectionFormData?: InspectionFormDataState } }
  | { type: typeof GET_INSPECTION_FORM_DATA_REQUEST }
  | { type: typeof GET_INSPECTION_FORM_DATA_SUCCESS; payload: Record<string, any> }
  | { type: typeof GET_INSPECTION_FORM_DATA_FAILED; payload: Record<string, any> }
  | { type: typeof SUBMIT_INSPECTION_REQUEST }
  | { type: typeof SUBMIT_INSPECTION_SUCCESS; payload: Record<string, any> }
  | { type: typeof SUBMIT_INSPECTION_FAILED; payload: Record<string, any> };

// =======================
// Initial States
// =======================
const initialFormDataState: InspectionFormDataState = {
  data: {},
  loading: false,
  error: {},
};

const initialSubmitDataState: InspectionFormDataState = {
  data: {},
  loading: false,
  error: {},
};

// =======================
// Reducers
// =======================
export const inspectionFormData = (
  state: InspectionFormDataState = initialFormDataState,
  action: InspectionFormAction
): InspectionFormDataState => {
  switch (action.type) {
    case REHYDRATE:
      return action.payload?.inspectionFormData ?? state;

    case GET_INSPECTION_FORM_DATA_REQUEST:
      return { ...state, loading: true, error: {} };

    case GET_INSPECTION_FORM_DATA_SUCCESS:
      return { ...state, data: action.payload ?? {}, loading: false, error: {} };

    case GET_INSPECTION_FORM_DATA_FAILED:
      return { ...state, loading: false, error: action.payload ?? {} };

    default:
      return state;
  }
};

export const inspectionSubmitData = (
  state: InspectionFormDataState = initialSubmitDataState,
  action: InspectionFormAction
): InspectionFormDataState => {
  switch (action.type) {
    case SUBMIT_INSPECTION_REQUEST:
      return { ...state, loading: true, error: {} };

    case SUBMIT_INSPECTION_SUCCESS:
      Toast.show({
        title: 'Inspection Submitted',
        description: 'Inspection submitted successfully!',
        placement: 'top',
        duration: 4000,
        variant: 'solid',
        backgroundColor: 'green.500',
      });
      return { ...state, data: action.payload ?? {}, loading: false, error: {} };

    case SUBMIT_INSPECTION_FAILED:
      Toast.show({
        title: 'Submission Failed',
        description: 'Something went wrong while submitting inspection.',
        placement: 'top',
        duration: 4000,
        variant: 'solid',
        backgroundColor: 'red.500',
      });
      return { ...state, loading: false, error: action.payload ?? {} };

    default:
      return state;
  }
};
