// reducers/checklistReducers.ts
import { Toast } from "native-base";
import { REHYDRATE } from "redux-persist";
import {
    GET_CHECKLIST_FORM_DATA_FAILED,
    GET_CHECKLIST_FORM_DATA_REQUEST,
    GET_CHECKLIST_FORM_DATA_SUCCESS,
    GET_CHECKLIST_SUBMISSIONS_DATA_REQUEST,
    GET_CHECKLIST_SUBMISSIONS_DATA_SUCCESS,
    GET_CHECKLIST_SUBMISSIONS_FAILED,
    SUBMIT_CHECKLIST_FAILED,
    SUBMIT_CHECKLIST_REQUEST,
    SUBMIT_CHECKLIST_SUCCESS,
} from "../actions/types";

// =======================
// Types
// =======================
export interface CheckListFormDataState {
  data: Record<string, any>;
  loading: boolean;
  error: Record<string, any>;
}

export interface CheckListSubmissionsState {
  data: any[];
  general_data: Record<string, any>;
  loading: boolean;
  error: Record<string, any>;
}

export interface ChecklistSubmitDataState {
  data: Record<string, any>;
  loading: boolean;
  error: Record<string, any>;
}

// =======================
// Action Types
// =======================
type CheckListFormAction =
  | { type: typeof REHYDRATE; payload?: { checkListFormData?: CheckListFormDataState } }
  | { type: typeof GET_CHECKLIST_FORM_DATA_REQUEST }
  | { type: typeof GET_CHECKLIST_FORM_DATA_SUCCESS; payload: Record<string, any> }
  | { type: typeof GET_CHECKLIST_FORM_DATA_FAILED; payload: Record<string, any> };

type CheckListSubmissionsAction =
  | { type: typeof GET_CHECKLIST_SUBMISSIONS_DATA_REQUEST }
  | { type: typeof GET_CHECKLIST_SUBMISSIONS_DATA_SUCCESS; payload: { submissions: any[]; general_data: Record<string, any> } }
  | { type: typeof GET_CHECKLIST_SUBMISSIONS_FAILED; payload: Record<string, any> };

type ChecklistSubmitAction =
  | { type: typeof SUBMIT_CHECKLIST_REQUEST }
  | { type: typeof SUBMIT_CHECKLIST_SUCCESS; payload: Record<string, any> }
  | { type: typeof SUBMIT_CHECKLIST_FAILED; payload: Record<string, any> };

// =======================
// Initial States
// =======================
const initialStateForm: CheckListFormDataState = {
  data: {},
  loading: false,
  error: {},
};

const initialStateSubmissions: CheckListSubmissionsState = {
  data: [],
  general_data: {},
  loading: false,
  error: {},
};

const initialStateSubmit: ChecklistSubmitDataState = {
  data: {},
  loading: false,
  error: {},
};

// =======================
// Reducers
// =======================
export const checkListFormData = (
  state: CheckListFormDataState = initialStateForm,
  action: CheckListFormAction
): CheckListFormDataState => {
  switch (action.type) {
    case REHYDRATE:
      return action.payload?.checkListFormData ?? state;

    case GET_CHECKLIST_FORM_DATA_REQUEST:
      return { ...state, loading: true, error: {} };

    case GET_CHECKLIST_FORM_DATA_SUCCESS:
      return { ...state, data: action.payload, loading: false, error: {} };

    case GET_CHECKLIST_FORM_DATA_FAILED:
      return { ...state, loading: false, error: action.payload };

    default:
      return state;
  }
};

export const checkListSubmissions = (
  state: CheckListSubmissionsState = initialStateSubmissions,
  action: CheckListSubmissionsAction
): CheckListSubmissionsState => {
  switch (action.type) {
    case GET_CHECKLIST_SUBMISSIONS_DATA_REQUEST:
      return { ...state, loading: true, error: {} };

    case GET_CHECKLIST_SUBMISSIONS_DATA_SUCCESS:
      return {
        ...state,
        data: action.payload.submissions ?? [],
        general_data: action.payload.general_data ?? {},
        loading: false,
        error: {},
      };

    case GET_CHECKLIST_SUBMISSIONS_FAILED:
      return { ...state, loading: false, error: action.payload };

    default:
      return state;
  }
};

export const checklistSubmitData = (
  state: ChecklistSubmitDataState = initialStateSubmit,
  action: ChecklistSubmitAction
): ChecklistSubmitDataState => {
  switch (action.type) {
    case SUBMIT_CHECKLIST_REQUEST:
      return { ...state, loading: true, error: {} };

    case SUBMIT_CHECKLIST_SUCCESS:
      Toast.show({
        title: "Checklist Submitted",
        placement: "top",
        duration: 4000,
        variant: "solid",
        description: "Checklist submitted successfully!",
        backgroundColor: "green.500",
      });
      return { ...state, data: action.payload, loading: false, error: {} };

    case SUBMIT_CHECKLIST_FAILED:
      Toast.show({
        title: "Failed to submit checklist",
        placement: "top",
        duration: 4000,
        variant: "solid",
        description: "Something went wrong while submitting checklist.",
        backgroundColor: "red.500",
      });
      return { ...state, loading: false, error: action.payload };

    default:
      return state;
  }
};
