# MAFGateway 2026

Fastify-based **modular monolith** in a simple monorepo. Easy to maintain now; straightforward to split into microservices later.

---

## Structure

```
MAF2.0/
├── apps/
│   ├── api/                 # HTTP API
│   │   └── src/
│   │       ├── index.ts     # Bootstrap, routes, load modules
│   │       ├── routes/      # Global: hello, health, jobs, metrics
│   │       └── modules/     # Domain modules (hello, auth, permits, …)
│   │           ├── hello/
│   │           ├── auth/
│   │           ├── permits/
│   │           └── ...
│   └── worker/              # Kafka consumer, background jobs
│       └── src/
│           └── handlers/
├── packages/
│   └── shared/              # Types, config, logger, Kafka client
├── scripts/                 # kafka-start.sh, kafka-stop.sh
├── docs/
├── package.json
└── pnpm-workspace.yaml
```

- **One shared package** – `packages/shared` has types, config, logger, and Kafka client.
- **Domain modules** – Each module is a folder under `apps/api/src/modules/` with an `index.ts` that exports `{ name, register }`. Add to `registry.ts` to load it.
- **Extract later** – Copy a module folder + shared into a new app and expose the same API over HTTP.

---

## Quick start

```bash
pnpm install
pnpm run build
pnpm run dev:api      # API (needs Kafka for job enqueue)
pnpm run dev:worker   # Worker (consumes from Kafka)
```

**Kafka (Docker):**

```bash
docker compose up -d zookeeper kafka
# .env: KAFKA_BROKERS=localhost:9092
```

**Kafka (host):** Use `./scripts/kafka-start.sh` and `./scripts/kafka-stop.sh` if Kafka is installed under `.kafka/`. See **docs/INSTALL-KAFKA.md** for full install options.

---

## Endpoints

| Endpoint | Description |
|----------|-------------|
| `GET /` | Links to hello module |
| `GET /hello` | Redirects to `/api/v1/hello` |
| `GET /api/v1/hello` | Hello module – Hello + directory structure (reference example) |
| `POST /api/v1/hello/job` | Enqueue `hello.demo` job (API → Kafka → Worker) |
| `GET /health` | Liveness |
| `GET /metrics` | Metrics (placeholder) |
| `POST /api/v1/jobs/enqueue` | Enqueue job (body: `{ "jobType": "report.generate", "payload": {} }`) |
| `GET /api/v1/auth/me`, `POST /api/v1/auth/login` | Auth module (demo) |

---

## Scripts

| Command | Description |
|---------|-------------|
| `pnpm run build` | Build all packages and apps |
| `pnpm run dev:api` | Run API in dev |
| `pnpm run dev:worker` | Run Worker in dev |
| `pnpm run dev:all` | Run API + Worker |
| `pnpm run typecheck` | Typecheck all |
| `./scripts/kafka-start.sh` | Start Zookeeper + Kafka (host install) |
| `./scripts/kafka-stop.sh` | Stop Kafka + Zookeeper |

---

## Hello module (reference example)

The **hello** module (`apps/api/src/modules/hello/`) is the template for using the folder structure with routes and jobs.

| Route | Method | Description |
|-------|--------|-------------|
| `GET /api/v1/hello` | GET | Returns "Hello World" + directory structure map |
| `POST /api/v1/hello/job` | POST | Enqueues a `hello.demo` job; the **worker** processes it |

**Flow: API → Kafka → Worker**

1. API – `POST /api/v1/hello/job` calls `app.kafka.publishJob(job)`.
2. Kafka – Job is produced to topic `mafgateway.jobs`.
3. Worker – Dispatches by `jobType` to `apps/worker/src/handlers/hello.ts`.

**Use as template for a new module**

1. Copy `apps/api/src/modules/hello/` to e.g. `permits/`; export `default` from the module (same as `permitsModule`).
2. In `apps/api/src/modules/manifest.ts`: add `'permits'` to the `MODULE_NAMES` array.
3. In `packages/shared/src/config.ts`: add `permits: { name: 'permits', enabled: true, basePath: '/api/v1/permits' }` to `config.modules`.
4. If the module enqueues jobs: add the payload type in `packages/shared/src/types.ts`, register a builder in `apps/api/src/routes/job-registry.ts`, add a handler in `apps/worker/src/handlers/` and register it in `handlers/index.ts`.

No new package or workspace entry—new folder, one name in the manifest, and one entry per job type in the registries.

---

## Other domain modules

- **Permits** – Add `apps/api/src/modules/permits/index.ts` exporting `permitsModule: ModuleDefinition` (same pattern as auth), then add to `registry.ts`.
- **Incidents** – Same pattern; add `incidentsModule` and register.
- **Analytics** – Same pattern; for heavy work, enqueue jobs via `app.kafka.publishJob()` and handle in the worker.
- **Notifications** – Same pattern; for async sending, enqueue jobs to the worker.

---

## Queue architecture

```
Clients / APIs → API Gateway → Producer Service → Message Broker (Kafka)
                                      │
                    ┌─────────────────┼─────────────────┬─────────────────┐
                    ▼                 ▼                 ▼                 ▼
              Worker Pool    Stream Processors    (retry/DLQ)    Dead Letter Queue
                    │                 │                                    │
                    ▼                 ▼                                    ▼
              ResultStore       ResultStore                          DLQ Processor
              (DB/Cache)        (DB/Cache)                           → ResultStore
```

- **API Gateway** – Fastify + gateway (request id, logging).
- **Producer Service** – Single path for enqueue; routes/modules use `app.producerService.enqueue(job)`.
- **Worker Pool** – Main consumer; heavy tasks; retry then DLQ.
- **Stream Processors** – Same topic, different group; stream-type jobs (e.g. analytics).
- **DLQ Processor** – Consumes DLQ topic; logs and stores for replay/alert.
- **ResultStore** – Sink for job results (in-memory by default; plug in DB/file).

See **docs/QUEUE-ARCHITECTURE.md** for details.

## Docs

- **docs/ARCHITECTURE.md** – Design, service boundaries, extracting to microservices.
- **docs/QUEUE-ARCHITECTURE.md** – Queue system (Gateway → Producer → Broker → Worker Pool / Stream / DLQ).
- **docs/HELLO-WORLD.md** – Hello module flow, request path, directory map.
- **docs/INSTALL-KAFKA.md** – Install Kafka (Docker or host).
- **docs/DEPLOYMENT-PLAN.md** – Azure deployment (API + Worker as separate App Services, Event Hubs).
