/**
 * MAFGateway 2026 - API Bootstrap
 * Modular monolith entry; each module is loaded and can be extracted to a microservice later.
 */

import Fastify from 'fastify';
import cors from '@fastify/cors';
import helmet from '@fastify/helmet';
import sensible from '@fastify/sensible';
import { loadConfig, createLogger, childLogger, createKafkaJobClient, createProducerService } from '@mafgateway/shared';
import { registerGateway } from './gateway.js';
import { registerModules } from './modules/registry.js';
import { registerRoutes } from './routes/index.js';

async function buildApp() {
  const config = loadConfig();
  const logger = createLogger({
    level: config.observability.logLevel,
    ...(config.env !== 'production' && {
      transport: {
        target: 'pino-pretty',
        options: { colorize: true },
      },
    }),
  });

  const app = Fastify({
    logger: false,
    requestIdHeader: 'x-request-id',
    requestIdLogLabel: 'requestId',
    trustProxy: true,
  });

  await app.register(cors, { origin: true });
  await app.register(helmet, { contentSecurityPolicy: false });
  await app.register(sensible);

  // API Gateway layer (request id, logging)
  await registerGateway(app);

  // Message Broker: Kafka client + Producer Service Layer (only path for enqueue)
  const kafkaClient = createKafkaJobClient(config.kafka);
  await kafkaClient.connect();
  app.decorate('kafka', kafkaClient);
  app.decorate('producerService', createProducerService(kafkaClient));
  app.addHook('onClose', async () => await kafkaClient.disconnect());

  await registerRoutes(app, config);
  await registerModules(app, config);

  app.setErrorHandler((err: unknown, request, reply) => {
    const req = request as { requestId?: string };
    const log = childLogger({
      module: 'api',
      requestId: req.requestId,
    });
    const error = err instanceof Error ? err : new Error(String(err));
    log.error({ err: error }, error.message);
    const statusCode = err && typeof err === 'object' && 'statusCode' in err ? Number((err as { statusCode?: number }).statusCode) : 500;
    const code = err && typeof err === 'object' && 'code' in err ? String((err as { code?: string }).code) : 'INTERNAL_ERROR';
    reply.status(statusCode).send({
      success: false,
      error: {
        code,
        message: config.env === 'production' ? 'Internal error' : error.message,
      },
      meta: { requestId: req.requestId },
    });
  });

  return { app, config, logger };
}

async function start() {
  const { app, config, logger } = await buildApp();
  try {
    await app.listen({ port: config.app.port, host: config.app.host });
    logger.info(
      { port: config.app.port, host: config.app.host, env: config.env },
      `${config.app.name} listening`
    );
  } catch (err) {
    logger.error(err);
    process.exit(1);
  }
}

start();
