/**
 * Hello World module – REFERENCE EXAMPLE for using the folder structure.
 *
 * This module shows:
 * 1. How to define a module (name + register)
 * 2. How to register routes under the module basePath
 * 3. How to enqueue a background job via app.kafka (API → Kafka → Worker)
 *
 * Copy this folder when creating a new domain module (e.g. permits, incidents).
 */

import type { FastifyInstance, FastifyRequest, FastifyReply } from 'fastify';
import type { ModuleDefinition, ModuleContext, ApiResponse, HelloDemoPayload } from '@mafgateway/shared';
import { childLogger } from '@mafgateway/shared';

const logger = childLogger({ module: 'hello' });

/** Route: GET /api/v1/hello – Hello message + directory structure reference */
async function getHello(request: FastifyRequest, reply: FastifyReply) {
  const req = request as FastifyRequest & { requestId?: string };
  return reply.send({
    success: true,
    data: {
      message: 'Hello World',
      from: 'hello module (reference example)',
      requestId: req.requestId,
      directoryStructure: {
      thisModule: 'apps/api/src/modules/hello/',
      flow: [
        '1. Request → apps/api/src/index.ts (bootstrap)',
        '2. Module loaded by apps/api/src/modules/registry.ts',
        '3. Routes registered here (apps/api/src/modules/hello/index.ts)',
        '4. Jobs enqueued via Producer Service → Message Broker → Worker Pool',
      ],
      paths: {
        apiBootstrap: 'apps/api/src/index.ts',
        moduleRegistry: 'apps/api/src/modules/registry.ts',
        thisFile: 'apps/api/src/modules/hello/index.ts',
        sharedPackage: 'packages/shared/ (types, config, logger, kafka)',
        workerHandlers: 'apps/worker/src/handlers/',
      },
    },
  },
  meta: { requestId: req.requestId },
  } as ApiResponse);
}

/** Route: POST /api/v1/hello/job – Enqueue a demo job (API → Kafka → Worker) */
async function postHelloJob(request: FastifyRequest, reply: FastifyReply, app: FastifyInstance) {
  const req = request as FastifyRequest & { requestId?: string };
  const body = (request.body as { message?: string }) ?? {};
  const jobId = crypto.randomUUID();

  const job: HelloDemoPayload = {
    jobId,
    jobType: 'hello.demo',
    correlationId: req.requestId,
    createdAt: new Date().toISOString(),
    message: body.message ?? 'Hello from the hello module',
    requestId: req.requestId,
  };

  try {
    await app.producerService.enqueue(job);
    logger.info({ jobId, message: job.message }, 'Hello demo job enqueued');
    return reply.send({
      success: true,
      data: {
        jobId,
        message: 'Job enqueued. Run the worker to process it: pnpm run dev:worker',
        jobType: 'hello.demo',
      },
      meta: { requestId: req.requestId },
    } as ApiResponse);
  } catch (err) {
    logger.error({ err }, 'Failed to enqueue hello job');
    return reply.code(503).send({
      success: false,
      error: { code: 'KAFKA_UNAVAILABLE', message: 'Could not enqueue job. Is Kafka running?' },
      meta: { requestId: req.requestId },
    } as ApiResponse);
  }
}

async function register(app: unknown, ctx: ModuleContext): Promise<void> {
  const fastify = app as FastifyInstance;
  const { basePath } = ctx;

  fastify.get(basePath, getHello);
  fastify.post(`${basePath}/job`, (req, reply) => postHelloJob(req, reply, fastify));

  logger.info({ basePath }, 'Hello module registered (reference example)');
}

const helloModule: ModuleDefinition = {
  name: 'hello',
  register,
};

export { helloModule };
export default helloModule;
