/**
 * Job payload builder registry – single place to add new enqueueable job types.
 * Add an entry here when you add a new job type; no need to edit the route handler.
 */

import type { JobPayload, ReportGeneratePayload, AnalyticsAggregatePayload } from '@mafgateway/shared';

export type JobBuilderContext = { requestId?: string };
export type JobBuilder = (
  customPayload: Record<string, unknown> | undefined,
  ctx: JobBuilderContext
) => JobPayload;

const builders: Partial<Record<JobPayload['jobType'], JobBuilder>> = {};

function registerJobBuilder<T extends JobPayload['jobType']>(
  jobType: T,
  builder: (payload: Record<string, unknown> | undefined, ctx: JobBuilderContext) => Extract<JobPayload, { jobType: T }>
): void {
  builders[jobType] = builder as JobBuilder;
}

/** Build base fields for all jobs */
function base(jobId: string, jobType: JobPayload['jobType'], ctx: JobBuilderContext, metadata?: Record<string, unknown>) {
  return {
    jobId,
    jobType,
    correlationId: ctx.requestId,
    createdAt: new Date().toISOString(),
    metadata,
  };
}

// ---- Register known job types (add new ones here) ----
registerJobBuilder('report.generate', (customPayload, ctx) => {
  const jobId = crypto.randomUUID();
  return {
    ...base(jobId, 'report.generate', ctx, customPayload),
    reportType: (customPayload?.reportType as string) ?? 'default',
    params: (customPayload?.params as Record<string, unknown>) ?? {},
    userId: customPayload?.userId as string | undefined,
  } as ReportGeneratePayload;
});

registerJobBuilder('analytics.aggregate', (customPayload, ctx) => {
  const jobId = crypto.randomUUID();
  return {
    ...base(jobId, 'analytics.aggregate', ctx, customPayload),
    period: (customPayload?.period as string) ?? 'day',
    dimensions: (customPayload?.dimensions as string[]) ?? [],
  } as AnalyticsAggregatePayload;
});

export function getJobBuilder(jobType: string): JobBuilder | undefined {
  return builders[jobType as JobPayload['jobType']];
}

export function getRegisteredJobTypes(): string[] {
  return Object.keys(builders);
}
