/**
 * Job enqueue API – uses job-registry for payload building. Add new job types in job-registry.ts.
 */

import type { FastifyInstance } from 'fastify';
import type { JobPayload } from '@mafgateway/shared';
import { getJobBuilder } from './job-registry.js';

interface EnqueueBody {
  jobType: string;
  payload?: Record<string, unknown>;
}

export async function registerJobRoutes(app: FastifyInstance): Promise<void> {
  app.post<{ Body: EnqueueBody }>('/api/v1/jobs/enqueue', async (request, reply) => {
    const { jobType, payload: customPayload } = request.body ?? {};
    const producer = app.producerService;
    if (!producer) return reply.code(503).send({ success: false, error: 'Jobs unavailable' });

    const builder = getJobBuilder(jobType);
    if (!builder) {
      return reply.code(400).send({ success: false, error: `Unknown jobType: ${jobType}` });
    }

    const req = request as { requestId?: string };
    const job = builder(customPayload, { requestId: req.requestId }) as JobPayload;
    await producer.enqueue(job);
    return reply.send({ success: true, data: { jobId: job.jobId } });
  });
}
