/**
 * Job handler registry – map jobType → handler. Add new entries here when adding job types.
 */

import type {
  JobPayload,
  HelloDemoPayload,
  ReportGeneratePayload,
  AnalyticsAggregatePayload,
  Logger,
} from '@mafgateway/shared';
import { helloDemoHandler } from './hello.js';
import { reportGenerateHandler } from './report.js';
import { analyticsAggregateHandler } from './analytics.js';

type JobHandler = (payload: JobPayload, logger: Logger) => Promise<void>;

const handlers: Partial<Record<JobPayload['jobType'], JobHandler>> = {
  'hello.demo': (p, log) => helloDemoHandler(p as HelloDemoPayload, log),
  'report.generate': (p, log) => reportGenerateHandler(p as ReportGeneratePayload, log),
  'analytics.aggregate': (p, log) => analyticsAggregateHandler(p as AnalyticsAggregatePayload, log),
};

export async function handleJob(payload: JobPayload, logger: Logger): Promise<void> {
  const handler = handlers[payload.jobType];
  if (!handler) {
    logger.warn({ jobType: payload.jobType }, 'Unknown job type, skipping');
    return;
  }
  await handler(payload, logger);
}
