# MAFGateway 2026 – Azure Deployment Plan

This plan keeps **heavy/async work in the Worker** and deploys API and Worker as **separate Azure services** so the worker can scale and fail independently.

---

## 1. Architecture Overview

```
                    ┌─────────────────────────────────────────────────────────────┐
                    │                      Azure                                  │
                    │  ┌──────────────┐     ┌─────────────────────────────────┐  │
                    │  │   API        │     │  Event Hubs (Kafka-compatible)   │  │
                    │  │ App Service  │────▶│  mafgateway.jobs / .dlq          │  │
                    │  │ or Container │     └──────────────┬──────────────────┘  │
                    │  │   App        │                    │                      │  │
                    │  └──────────────┘                    ▼                      │  │
                    │         │                  ┌─────────────────────────────────┐
                    │         │                  │  Worker (heavy tasks)           │  │
                    │         │                  │  Option A: App Service /         │  │
                    │         │                  │  Container App (recommended)     │  │
                    │         │                  │  Option B: Azure Functions      │  │
                    │         │                  │  (Event Hubs trigger)            │  │
                    │         │                  └─────────────────────────────────┘  │
                    │         │                                                      │  │
                    │         ▼                                                      │  │
                    │  ┌──────────────┐     ┌──────────────┐  ┌──────────────────┐  │  │
                    │  │ App Insights │     │ Key Vault    │  │ Storage (files,  │  │  │
                    │  │ Log Analytics│     │ (secrets)    │  │ reports, DLQ)    │  │  │
                    │  └──────────────┘     └──────────────┘  └──────────────────┘  │  │
                    └─────────────────────────────────────────────────────────────┘
```

- **API**: Stateless HTTP; scale on request load.
- **Worker**: Consumes from Event Hubs; runs report generation, analytics aggregation, file processing. Scale by instance count and/or partition count.
- **Heavy tasks** stay in the Worker only; API only enqueues jobs.

---

## 2. Recommended Option: App Service (API) + App Service / Container App (Worker)

### 2.1 API – Azure App Service (or Container App)

| Item | Recommendation |
|------|----------------|
| **Service** | Azure App Service (Linux, Node 20) or Azure Container Apps |
| **Scaling** | Horizontal: 1–N instances; scale out on CPU/memory or HTTP queue |
| **Deploy** | Docker image from ACR (Azure Container Registry) or Oryx build |

- **Why separate**: API stays lightweight (enqueue only); no long-running or CPU-heavy work.
- **Config**: Set env vars (see Section 5); use Key Vault references for secrets (Event Hubs connection string, etc.).

### 2.2 Worker – Azure App Service or Container App (recommended for heavy tasks)

| Item | Recommendation |
|------|----------------|
| **Service** | **Azure App Service** (Linux, Node 20, always-on) or **Azure Container App** (scale to zero possible) |
| **Scaling** | Scale out by **instance count**; partition count in Event Hubs defines max parallelism. |
| **Heavy tasks** | Report generation, analytics aggregation, file processing run **only** here. |

- **Why separate**: Worker can have more CPU/memory, different scale rules, and can be scaled to 0 when using Container Apps if no jobs.
- **Deploy**: Same repo, different Dockerfile (`Dockerfile.worker`) and image tag (e.g. `mafgateway-worker:latest`).

### 2.3 Alternative: Worker as Azure Functions (Event Hubs trigger)

Use when you prefer serverless and per-invocation billing:

| Item | Detail |
|------|--------|
| **Trigger** | Event Hubs trigger on `mafgateway.jobs` (and optionally DLQ). |
| **Runtime** | Node 20; one function per job type or one function that dispatches by `jobType`. |
| **Heavy tasks** | Each message runs in a function; use durable functions or queue to another service for very long jobs. |

- **Trade-off**: Time limit per function (e.g. 5–10 min); for very heavy/long jobs, keep a dedicated Worker App Service or Container App.

---

## 3. Messaging: Azure Event Hubs (Kafka-compatible)

Use **Event Hubs** with Kafka protocol so existing Kafka client code stays the same.

| Step | Action |
|------|--------|
| 1 | Create Event Hubs namespace with **Kafka enabled**. |
| 2 | Create two Event Hubs: `mafgateway.jobs` (main), `mafgateway.jobs.dlq` (dead-letter). |
| 3 | Set **connection string** or **SAS** in Key Vault; give API produce access, Worker consume access. |
| 4 | Configure **KAFKA_BROKERS** = `&lt;namespace&gt;.servicebus.windows.net:9093` and SSL. |

**Env for Event Hubs (Kafka):**

- `KAFKA_BROKERS=<namespace>.servicebus.windows.net:9093`
- `KAFKA_SASL_MECHANISM=plain`
- `KAFKA_SECURITY_PROTOCOL=SASL_SSL`
- SASL username = `$ConnectionString`, password = Event Hubs connection string (or use Azure identity later).

If you add SASL/SSL to the shared Kafka client, the same codebase works against Event Hubs in Azure and Kafka locally.

---

## 4. Deployment Topology Summary

| Component | Azure service | Responsibility |
|-----------|--------------|----------------|
| **API** | App Service or Container App | HTTP API; enqueue jobs only; no heavy work. |
| **Worker** | App Service or Container App (or Functions) | Consume from Event Hubs; run **all** heavy tasks (reports, analytics, file processing). |
| **Messaging** | Event Hubs (Kafka) | `mafgateway.jobs`, `mafgateway.jobs.dlq`. |
| **Secrets** | Key Vault | Event Hubs connection string, DB credentials, etc. |
| **Observability** | App Insights + Log Analytics | Logs, traces, metrics from API and Worker. |

---

## 5. Environment Variables by Service

### API (App Service / Container App)

```bash
NODE_ENV=production
APP_NAME=MAFGateway-2026
PORT=8080
HOST=0.0.0.0

# Event Hubs (Kafka)
KAFKA_BROKERS=<namespace>.servicebus.windows.net:9093
KAFKA_CLIENT_ID=mafgateway-api
KAFKA_TOPIC_JOBS=mafgateway.jobs
KAFKA_TOPIC_DLQ=mafgateway.jobs.dlq
# If using SASL_SSL (Event Hubs):
# KAFKA_SASL_MECHANISM=plain
# KAFKA_SECURITY_PROTOCOL=SASL_SSL
# KAFKA_SASL_USERNAME=$ConnectionString
# KAFKA_SASL_PASSWORD=<connection-string-from-keyvault>

LOG_LEVEL=info
```

- **Worker** does **not** need to be in the same App Service plan; it is a **separate** App Service or Container App.

### Worker (App Service / Container App)

```bash
NODE_ENV=production
APP_NAME=MAFGateway-2026-Worker

KAFKA_BROKERS=<namespace>.servicebus.windows.net:9093
KAFKA_CLIENT_ID=mafgateway-worker
KAFKA_CONSUMER_GROUP=mafgateway-workers
KAFKA_TOPIC_JOBS=mafgateway.jobs
KAFKA_TOPIC_DLQ=mafgateway.jobs.dlq
# Same SASL/SSL vars if required

LOG_LEVEL=info
```

- Store secrets (e.g. `KAFKA_SASL_PASSWORD`, DB URLs) in **Key Vault** and reference from App Service/Container App.

---

## 6. CI/CD (GitHub Actions Example)

- **Build** once per commit; build both API and Worker images.
- **Push** to Azure Container Registry (ACR).
- **Deploy** API to API App Service (or Container App); deploy Worker to **separate** Worker App Service (or Container App).

Example flow:

1. **Build & push**
   - Build `Dockerfile.api` → `acr.azurecr.io/mafgateway-api:${SHA}`
   - Build `Dockerfile.worker` → `acr.azurecr.io/mafgateway-worker:${SHA}`

2. **Deploy API**
   - Update API App Service (or Container App) to use `mafgateway-api:${SHA}`.

3. **Deploy Worker**
   - Update Worker App Service (or Container App) to use `mafgateway-worker:${SHA}`.

4. **Optional**
   - Deploy API and Worker in one pipeline with two jobs; or separate pipelines for independent releases.

---

## 7. Scaling and Heavy-Task Guarantees

| Goal | How |
|------|-----|
| **Heavy work only in Worker** | API code only calls `app.kafka.publishJob(...)`; no report/analytics/file logic in API. |
| **Worker scales with backlog** | Scale Worker App Service/Container App by **custom metric** (e.g. Event Hubs backlog length) or by instance count. |
| **Partition count** | Set Event Hubs partition count (e.g. 8–32) so multiple Worker instances can consume in parallel. |
| **DLQ** | Failed jobs after retries go to `mafgateway.jobs.dlq`; optional Azure Function or second Worker to process/alerts. |

---

## 8. Checklist

- [ ] Create resource group and Azure Container Registry (ACR).
- [ ] Create Event Hubs namespace + hubs: `mafgateway.jobs`, `mafgateway.jobs.dlq`.
- [ ] Create **API** App Service or Container App; configure env and Key Vault; deploy API image.
- [ ] Create **Worker** App Service or Container App (separate from API); configure env and Key Vault; deploy Worker image.
- [ ] Ensure Worker only runs heavy tasks (reports, analytics, file processing); API only enqueues.
- [ ] Configure scaling (instance count, optional Event Hubs backlog metric).
- [ ] Wire App Insights and Log Analytics for both API and Worker.
- [ ] Add CI/CD to build both images and deploy to respective Azure services.

Using this plan, the **worker handles heavy tasks in separate Azure App Services (or Container Apps)**, and optionally in Azure Functions, while the API stays lightweight and only enqueues jobs.
