/** Kafka producer/consumer with retry and DLQ */

import { Kafka, type Producer, type Consumer, type EachMessagePayload } from 'kafkajs';
import type { JobPayload } from './types.js';
import { childLogger } from './logger.js';

const logger = childLogger({ module: 'kafka' });
const MAX_RETRIES = 5;
const RETRY_BACKOFF_MS = 1000;

export interface KafkaClientConfig {
  brokers: string[];
  clientId: string;
  consumerGroup?: string;
  topics: { jobs: string; dlq: string };
  sasl?: { mechanism: 'plain'; username: string; password: string };
  ssl?: boolean;
}

export class KafkaJobClient {
  private kafka: Kafka;
  private producer: Producer | null = null;
  private consumer: Consumer | null = null;
  private config: KafkaClientConfig;
  private messageHandler: ((payload: JobPayload) => Promise<void>) | null = null;

  constructor(config: KafkaClientConfig) {
    this.config = config;
    this.kafka = new Kafka({
      clientId: config.clientId,
      brokers: config.brokers,
      ssl: config.ssl ?? false,
      sasl: config.sasl,
      retry: { retries: 5, initialRetryTime: 300, maxRetryTime: 30000 },
    });
  }

  async connect(): Promise<void> {
    this.producer = this.kafka.producer();
    await this.producer.connect();
    logger.info('Kafka producer connected');
  }

  async disconnect(): Promise<void> {
    if (this.producer) await this.producer.disconnect();
    if (this.consumer) await this.consumer.disconnect();
    this.producer = null;
    this.consumer = null;
    logger.info('Kafka client disconnected');
  }

  async publishJob(job: JobPayload, key?: string): Promise<void> {
    if (!this.producer) throw new Error('Producer not connected');
    await this.producer.send({
      topic: this.config.topics.jobs,
      messages: [{
        key: key ?? job.jobId,
        value: JSON.stringify(job),
        headers: { 'content-type': 'application/json', 'job-type': job.jobType, ...(job.correlationId ? { 'correlation-id': job.correlationId } : {}) },
      }],
    });
    logger.info({ jobId: job.jobId, jobType: job.jobType }, 'Job published');
  }

  async publishToDLQ(job: JobPayload, error: Error): Promise<void> {
    if (!this.producer) throw new Error('Producer not connected');
    await this.producer.send({
      topic: this.config.topics.dlq,
      messages: [{ key: job.jobId, value: JSON.stringify({ ...job, dlqReason: error.message, dlqAt: new Date().toISOString() }) }],
    });
    logger.warn({ jobId: job.jobId, error: error.message }, 'Job sent to DLQ');
  }

  async startConsumer(groupId: string, handler: (payload: JobPayload) => Promise<void>): Promise<void> {
    this.messageHandler = handler;
    this.consumer = this.kafka.consumer({ groupId });
    await this.consumer.connect();
    await this.consumer.subscribe({ topic: this.config.topics.jobs, fromBeginning: false });
    await this.consumer.run({
      eachMessage: async (payload: EachMessagePayload) => {
        const value = payload.message.value?.toString();
        if (!value || !this.messageHandler) return;
        let job: JobPayload;
        try {
          job = JSON.parse(value) as JobPayload;
        } catch {
          logger.error({ topic: payload.topic }, 'Invalid job JSON');
          return;
        }
        const retryCount = (job.retryCount ?? 0) + 1;
        try {
          await this.messageHandler(job);
        } catch (error) {
          const err = error instanceof Error ? error : new Error(String(error));
          if (retryCount >= MAX_RETRIES) await this.publishToDLQ({ ...job, retryCount }, err);
          else {
            const backoff = RETRY_BACKOFF_MS * Math.pow(2, retryCount);
            logger.warn({ jobId: job.jobId, retryCount, backoffMs: backoff }, 'Job failed, will retry');
            await new Promise((r) => setTimeout(r, backoff));
            await this.publishJob({ ...job, retryCount });
          }
        }
      },
    });
    logger.info({ groupId, topic: this.config.topics.jobs }, 'Consumer started');
  }
}

export function createKafkaJobClient(config: KafkaClientConfig): KafkaJobClient {
  return new KafkaJobClient(config);
}
