/** Shared types and contracts – used by API and Worker */

export interface ApiResponse<T = unknown> {
  success: boolean;
  data?: T;
  error?: { code: string; message: string };
  meta?: { requestId?: string; timestamp?: string };
}

export interface PaginatedResponse<T> extends ApiResponse<T[]> {
  meta: ApiResponse['meta'] & { page: number; limit: number; total: number; totalPages: number };
}

export type JobType =
  | 'hello.demo'
  | 'report.generate'
  | 'analytics.aggregate'
  | 'file.process'
  | 'notification.send'
  | 'incident.sync';

export interface BaseJobPayload {
  jobId: string;
  jobType: JobType;
  correlationId?: string;
  createdAt: string;
  retryCount?: number;
  metadata?: Record<string, unknown>;
}

export interface HelloDemoPayload extends BaseJobPayload {
  jobType: 'hello.demo';
  message?: string;
  requestId?: string;
}

export interface ReportGeneratePayload extends BaseJobPayload {
  jobType: 'report.generate';
  reportType: string;
  params: Record<string, unknown>;
  userId?: string;
}

export interface AnalyticsAggregatePayload extends BaseJobPayload {
  jobType: 'analytics.aggregate';
  period: string;
  dimensions?: string[];
}

export interface FileProcessPayload extends BaseJobPayload {
  jobType: 'file.process';
  fileKey: string;
  operation: 'validate' | 'transform' | 'import';
}

export type JobPayload =
  | HelloDemoPayload
  | ReportGeneratePayload
  | AnalyticsAggregatePayload
  | FileProcessPayload
  | (BaseJobPayload & { jobType: 'notification.send' | 'incident.sync'; [k: string]: unknown });

export interface ModuleConfig {
  name: string;
  enabled: boolean;
  basePath?: string;
  [key: string]: unknown;
}

export interface HealthCheckResult {
  status: 'ok' | 'degraded' | 'down';
  module: string;
  latencyMs?: number;
  details?: Record<string, unknown>;
}

export interface ModuleContext {
  basePath: string;
  config: ModuleConfig;
}

export interface ModuleDefinition {
  name: string;
  register: (app: unknown, ctx: ModuleContext) => Promise<void>;
}
