/**
 * Auth module – lives inside API. To extract: copy this folder + shared to a new service.
 */

import type { FastifyInstance, FastifyRequest, FastifyReply } from 'fastify';
import type { ModuleDefinition, ModuleContext, ApiResponse } from '@mafgateway/shared';
import { childLogger } from '@mafgateway/shared';

const logger = childLogger({ module: 'auth' });

async function authRoutes(app: FastifyInstance, basePath: string) {
  app.get(`${basePath}/me`, async (request: FastifyRequest, reply: FastifyReply) => {
    const req = request as FastifyRequest & { requestId?: string };
    const response: ApiResponse<{ userId: string; scope: string[] }> = {
      success: true,
      data: { userId: 'demo-user', scope: ['read:profile'] },
      meta: { requestId: req.requestId, timestamp: new Date().toISOString() },
    };
    return reply.send(response);
  });

  app.post(`${basePath}/login`, async (request: FastifyRequest, reply: FastifyReply) => {
    const body = request.body as { username?: string; password?: string };
    if (!body?.username) {
      return reply.code(400).send({
        success: false,
        error: { code: 'VALIDATION_ERROR', message: 'username required' },
      } as ApiResponse);
    }
    logger.info({ username: body.username }, 'Login attempt');
    return reply.send({
      success: true,
      data: { token: 'demo-jwt', expiresIn: 3600 },
    } as ApiResponse);
  });
}

async function register(app: unknown, ctx: ModuleContext): Promise<void> {
  await authRoutes(app as FastifyInstance, ctx.basePath);
  logger.info({ basePath: ctx.basePath }, 'Auth module registered');
}

const authModule: ModuleDefinition = {
  name: 'auth',
  register,
};

export { authModule };
export default authModule;
