/**
 * Loads domain modules by convention: each name in MODULE_NAMES is dynamic-imported
 * from ./<name>/index.js (must export default ModuleDefinition).
 * To add a module: create modules/<name>/index.ts and add <name> to manifest.ts.
 */

import type { FastifyInstance } from 'fastify';
import type { AppConfig, ModuleDefinition } from '@mafgateway/shared';
import { MODULE_NAMES } from './manifest.js';

export async function registerModules(app: FastifyInstance, config: AppConfig): Promise<void> {
  for (const name of MODULE_NAMES) {
    const moduleConfig = config.modules[name];
    if (!moduleConfig?.enabled) continue;
    const mod = await import(`./${name}/index.js`);
    const definition = mod.default as ModuleDefinition;
    if (!definition?.register) {
      app.log?.warn?.({ module: name }, 'Module has no default export with register, skipping');
      continue;
    }
    await definition.register(app, {
      basePath: moduleConfig.basePath ?? `/api/v1/${name}`,
      config: moduleConfig,
    });
    app.log?.info?.({ module: definition.name, basePath: moduleConfig.basePath }, 'Module registered');
  }
}
