/**
 * Dead Letter Queue Processor – consumes from DLQ topic.
 * Logs and persists failed messages for inspection / replay / alerting.
 */

import type { JobPayload, JobResult, ResultStore } from '@mafgateway/shared';
import type { Logger } from '@mafgateway/shared';

interface DLQMessage {
  jobId: string;
  jobType: string;
  dlqReason?: string;
  dlqAt?: string;
  [k: string]: unknown;
}

export async function handleDLQMessage(
  msg: { key: string | null; value: string },
  logger: Logger,
  resultStore: ResultStore
): Promise<void> {
  let payload: DLQMessage;
  try {
    payload = JSON.parse(msg.value) as DLQMessage;
  } catch {
    logger.error({ value: msg.value.slice(0, 200) }, 'DLQ Processor: invalid JSON');
    return;
  }
  logger.warn(
    { jobId: payload.jobId, jobType: payload.jobType, dlqReason: payload.dlqReason, dlqAt: payload.dlqAt },
    'DLQ Processor: received failed job'
  );
  const result: JobResult = {
    jobId: payload.jobId,
    jobType: payload.jobType ?? 'unknown',
    status: 'failed',
    error: payload.dlqReason ?? 'Unknown',
    completedAt: payload.dlqAt ?? new Date().toISOString(),
  };
  await resultStore.save(payload.jobId, payload as JobPayload, result);
}
