/**
 * MAFGateway 2026 – Worker process.
 *
 * Queue architecture:
 *   Message Broker (Kafka)
 *        │
 *   ┌────┴────┬─────────────────┬─────────────────┐
 *   ▼         ▼                 ▼                 ▼
 * Worker Pool   Stream Processors   (DLQ topic)   Dead Letter Queue
 *   │                 │                              │
 *   ▼                 ▼                              ▼
 * ResultStore     ResultStore                    DLQ Processor
 * (DB/Cache)      (DB/Cache)                     → ResultStore / alert
 */

import {
  loadConfig,
  createLogger,
  createKafkaJobClient,
  createQueueConsumer,
  createDefaultResultStore,
  type JobPayload,
  type JobResult,
  } from '@mafgateway/shared';
import { handleJob } from './handlers/index.js';
import { handleStreamMessage } from './stream-processor.js';
import { handleDLQMessage } from './dlq-processor.js';

async function main() {
  const config = loadConfig();
  const logger = createLogger({ level: config.observability.logLevel });

  // Sink: Database / Cache / Storage (Worker Pool + Stream + DLQ write here)
  const resultStore = createDefaultResultStore();

  // ---- Worker Pool: consumes main queue, heavy tasks, retry + DLQ ----
  const kafkaPool = createKafkaJobClient({
    ...config.kafka,
    clientId: config.kafka.clientId + '-pool',
  });
  await kafkaPool.connect();

  await kafkaPool.startConsumer(config.kafka.consumerGroup, async (payload: JobPayload) => {
    // Stream Processors handle these; Worker Pool skips to avoid duplicate work
    if (payload.jobType === 'analytics.aggregate') return;

    logger.info({ jobId: payload.jobId, jobType: payload.jobType }, 'Worker Pool: processing');
    try {
      await handleJob(payload, logger);
      const result: JobResult = {
        jobId: payload.jobId,
        jobType: payload.jobType,
        status: 'completed',
        completedAt: new Date().toISOString(),
      };
      await resultStore.save(payload.jobId, payload, result);
    } catch (err) {
      const error = err instanceof Error ? err : new Error(String(err));
      const result: JobResult = {
        jobId: payload.jobId,
        jobType: payload.jobType,
        status: 'failed',
        error: error.message,
        completedAt: new Date().toISOString(),
      };
      await resultStore.save(payload.jobId, payload, result);
      throw err; // let KafkaJobClient handle retry/DLQ
    }
  });

  // ---- Stream Processors: same topic, different group, stream-type jobs only ----
  const streamConsumer = createQueueConsumer(
    {
      brokers: config.kafka.brokers,
      clientId: config.kafka.clientId + '-stream',
      sasl: config.kafka.sasl,
      ssl: config.kafka.ssl,
    },
    config.kafka.streamProcessorGroup,
    config.kafka.topics.jobs,
    async (msg) => {
      await handleStreamMessage(msg, logger, resultStore);
    }
  );
  await streamConsumer.start();

  // ---- Dead Letter Queue Processor: consumes DLQ topic ----
  const dlqConsumer = createQueueConsumer(
    {
      brokers: config.kafka.brokers,
      clientId: config.kafka.clientId + '-dlq',
      sasl: config.kafka.sasl,
      ssl: config.kafka.ssl,
    },
    config.kafka.dlqProcessorGroup,
    config.kafka.topics.dlq,
    async (msg) => {
      await handleDLQMessage(msg, logger, resultStore);
    }
  );
  await dlqConsumer.start();

  const shutdown = async () => {
    logger.info('Shutting down worker...');
    await streamConsumer.disconnect();
    await dlqConsumer.disconnect();
    await kafkaPool.disconnect();
    process.exit(0);
  };
  process.on('SIGTERM', shutdown);
  process.on('SIGINT', shutdown);

  logger.info(
    {
      poolGroup: config.kafka.consumerGroup,
      streamGroup: config.kafka.streamProcessorGroup,
      dlqGroup: config.kafka.dlqProcessorGroup,
    },
    'Worker started (Worker Pool + Stream Processors + DLQ Processor)'
  );
}

main().catch((err) => {
  console.error(err);
  process.exit(1);
});
