/**
 * Stream Processors – consume from main queue (same topic as Worker Pool, different group).
 * Handle only "stream" job types (e.g. analytics.aggregate); others are skipped (Worker Pool handles them).
 */

import type { JobPayload, JobResult, ResultStore } from '@mafgateway/shared';
import type { Logger } from '@mafgateway/shared';
import { analyticsAggregateHandler } from './handlers/analytics.js';

const STREAM_JOB_TYPES: JobPayload['jobType'][] = ['analytics.aggregate'];

export async function handleStreamMessage(
  msg: { key: string | null; value: string },
  logger: Logger,
  resultStore: ResultStore
): Promise<void> {
  let job: JobPayload;
  try {
    job = JSON.parse(msg.value) as JobPayload;
  } catch {
    return;
  }
  if (!STREAM_JOB_TYPES.includes(job.jobType)) {
    return; // Worker Pool handles this
  }
  logger.info({ jobId: job.jobId, jobType: job.jobType }, 'Stream Processor: processing');
  try {
    await analyticsAggregateHandler(job as import('@mafgateway/shared').AnalyticsAggregatePayload, logger);
    const result: JobResult = {
      jobId: job.jobId,
      jobType: job.jobType,
      status: 'completed',
      completedAt: new Date().toISOString(),
    };
    await resultStore.save(job.jobId, job, result);
  } catch (err) {
    const error = err instanceof Error ? err : new Error(String(err));
    logger.warn({ jobId: job.jobId, error: error.message }, 'Stream Processor: failed');
    const result: JobResult = {
      jobId: job.jobId,
      jobType: job.jobType,
      status: 'failed',
      error: error.message,
      completedAt: new Date().toISOString(),
    };
    await resultStore.save(job.jobId, job, result);
  }
}
