# Hello World module – reference example

The **hello** module is the reference for how to use the folder structure: routes, jobs, and workers.

## Try it

```bash
# Terminal 1: API
pnpm run dev:api

# Terminal 2: Worker (optional; needed to process enqueued jobs)
pnpm run dev:worker

# Terminal 3: requests
curl http://localhost:3000/
curl http://localhost:3000/hello          # redirects to /api/v1/hello
curl http://localhost:3000/api/v1/hello  # Hello + directory structure
curl -X POST http://localhost:3000/api/v1/hello/job -H "Content-Type: application/json" -d '{"message":"My first job"}'
```

With the worker running, the last request enqueues a job; the worker logs that it processed the `hello.demo` job.

---

## What the hello module demonstrates

| Piece | Location | Purpose |
|-------|----------|---------|
| **Module definition** | `apps/api/src/modules/hello/index.ts` | Exports `helloModule` with `name` + `register(app, ctx)`. |
| **Routes** | Same file | `GET /api/v1/hello` (hello + structure), `POST /api/v1/hello/job` (enqueue job). |
| **Job enqueue** | Same file, `postHelloJob` | Uses `app.kafka.publishJob(job)` (API → Kafka). |
| **Shared types** | `packages/shared/src/types.ts` | `JobType` + `HelloDemoPayload` for `hello.demo` jobs. |
| **Config** | `packages/shared/src/config.ts` | `modules.hello` with `basePath: '/api/v1/hello'`. |
| **Registry** | `apps/api/src/modules/registry.ts` | Imports `helloModule` and adds it to the `modules` array. |
| **Worker handler** | `apps/worker/src/handlers/hello.ts` | Handles `hello.demo` (Kafka → Worker). |
| **Worker dispatch** | `apps/worker/src/handlers/index.ts` | `case 'hello.demo':` → `helloDemoHandler`. |

---

## Flow: API → Kafka → Worker

```
POST /api/v1/hello/job
    │
    ▼
apps/api/src/modules/hello/index.ts  →  app.kafka.publishJob(job)
    │
    ▼
Kafka topic: mafgateway.jobs
    │
    ▼
apps/worker/src/index.ts  (consumer)
    │
    ▼
apps/worker/src/handlers/index.ts  →  case 'hello.demo'
    │
    ▼
apps/worker/src/handlers/hello.ts  →  helloDemoHandler()
```

---

## Using the hello module as a template

1. **New module (e.g. permits)**  
   Copy `apps/api/src/modules/hello/` to `apps/api/src/modules/permits/`.  
   Rename to `permitsModule`, routes under `/api/v1/permits`.  
   Add to `registry.ts` and `config.modules.permits`.

2. **New job type**  
   In `packages/shared/src/types.ts`: add to `JobType` and define a payload (e.g. `PermitsSyncPayload`).  
   In `apps/worker/src/handlers/`: add a handler and a `case` in `handlers/index.ts`.

3. **Enqueue from a module**  
   In the module’s route handler, use `app.kafka.publishJob(payload)`.  
   Ensure the payload matches a `JobType` and that the worker has a handler for it.

See also: **apps/api/src/modules/hello/README.md**.

---

## Directory layout (quick ref)

| Path | Purpose |
|------|--------|
| **apps/api/src/index.ts** | Bootstrap, plugins, routes, Kafka, module loader. |
| **apps/api/src/routes/** | Global routes (/, /health, /api/v1/jobs/enqueue, /metrics). |
| **apps/api/src/modules/hello/** | Hello module (reference): routes + job enqueue. |
| **apps/api/src/modules/registry.ts** | Loads all domain modules. |
| **packages/shared/** | Types, config, logger, Kafka client. |
| **apps/worker/src/handlers/** | Job handlers by `jobType`. |
