# MAF2.0 – Code & Folder Structure Review (Scalability)

## Current structure (summary)

| Layer | Location | Notes |
|-------|----------|--------|
| **API** | `apps/api/src/` | Bootstrap, gateway, routes, module registry |
| **Worker** | `apps/worker/src/` | Kafka consumers (pool, stream, DLQ), job handlers |
| **Shared** | `packages/shared/` | Types, config, logger, Kafka, ProducerService, ResultStore |
| **Domain modules** | `apps/api/src/modules/` | hello, auth (used by API) |
| **Root modules/** | `modules/` | auth (built), analytics/incidents/permits/notifications (empty) – **not in pnpm workspace** |

---

## What’s working well

1. **Single shared package** – One place for types, config, logger, Kafka. Reduces coupling and version drift.
2. **Module contract** – `ModuleDefinition` (`name`, `register(app, ctx)`) is clear and extractable to a microservice later.
3. **Producer Service** – Single path for enqueue (`app.producerService.enqueue`) keeps broker usage consistent.
4. **Worker architecture** – Pool + stream processors + DLQ is scalable (more instances, same consumer groups).
5. **Config-driven modules** – `config.modules[name].enabled` allows turning modules off without code change.
6. **Fastify typing** – `fastify.d.ts` augments `FastifyInstance` with `kafka` and `producerService`.

---

## Scalability issues and recommendations

### 1. Duplicate / confusing module locations

- **Issue:** Domain modules live in `apps/api/src/modules/` (hello, auth). Root `modules/` has auth (with dist) and empty folders; it’s outside the pnpm workspace and not loaded by the API.
- **Recommendation:** Treat **`apps/api/src/modules/`** as the single source of truth. Remove or repurpose root `modules/` (e.g. delete empty ones, or move `modules/auth` into `apps/api/src/modules/` if it’s meant to be the same and remove the duplicate).

### 2. Manual module registry

- **Issue:** Adding a module requires editing `registry.ts` (import + add to array) and `config.ts` (add to `config.modules`). Easy to forget one.
- **Recommendation:** Use **convention-based discovery**: e.g. all folders under `src/modules/` that export a `ModuleDefinition` are loaded. Config can stay as-is for `enabled`/`basePath`, or be derived from a single manifest.

### 3. Job type handling in multiple places

- **Issue:** New job types require changes in: `packages/shared/src/types.ts`, `apps/api/src/routes/jobs.ts` (enqueue body parsing), `apps/worker/src/handlers/index.ts` (dispatch), and optionally `stream-processor.ts` for stream jobs.
- **Recommendation:** Introduce a **job registry**: one place that maps `jobType` → payload schema/validator and (on worker) handler. Enqueue route and worker dispatch both use this registry so adding a job type = one registration.

### 4. Route registration in bootstrap

- **Issue:** `index.ts` explicitly registers health, hello, jobs, metrics, then `registerModules()`. Fine for now, but as you add more global routes it gets noisy.
- **Recommendation:** Group core routes (health, metrics, jobs) in a small **routes registry** or `routes/index.ts` that `index.ts` calls once. Keeps bootstrap minimal and scales with more route groups.

### 5. ResultStore for horizontal scaling

- **Issue:** `createDefaultResultStore()` is in-memory; multiple API/worker instances don’t share state.
- **Recommendation:** Keep the interface; add a **Redis or DB-backed ResultStore** and choose it via config (e.g. `RESULT_STORE=redis`). No change to callers.

### 6. Deep health checks

- **Issue:** `GET /health?deep=true` has a placeholder; modules don’t register health checks.
- **Recommendation:** Extend `ModuleContext` or module contract with an optional `healthCheck?(): Promise<HealthCheckResult>`. Registry runs these when `deep=true` and merges into the response.

### 7. Unused packages (types, config, logger, kafka-client)

- **Issue:** `packages/` contains types, config, logger, kafka-client (with dist); apps use only `@mafgateway/shared`. Docs say you consolidated to one shared package.
- **Recommendation:** If you’re committed to a single shared package, remove or archive the unused packages to avoid confusion. If you prefer to split later, add them to the workspace and migrate shared to use them step by step.

### 8. Fastify instance augmentation

- **Issue:** As you add more shared services (DB, cache, etc.), `fastify.d.ts` will grow with many top-level props.
- **Recommendation:** Optional: add a single `app.services` (or `app.context`) object and augment that instead, so one place for “shared services” and fewer type edits.

---

## Recommended folder structure (scalable)

```
MAF2.0/
├── apps/
│   ├── api/
│   │   └── src/
│   │       ├── index.ts              # Bootstrap only: load config, create app, register gateway, routes, modules
│   │       ├── gateway.ts
│   │       ├── routes/               # Global routes (health, metrics, jobs) + routes/index.ts that registers all
│   │       ├── modules/              # Single place for domain modules
│   │       │   ├── hello/
│   │       │   ├── auth/
│   │       │   ├── permits/          # Add when needed
│   │       │   └── ...
│   │       └── types/
│   └── worker/
│       └── src/
│           ├── index.ts
│           ├── handlers/             # One file per job type + registry (map jobType → handler)
│           ├── stream-processor.ts
│           └── dlq-processor.ts
├── packages/
│   └── shared/
│       └── src/
│           ├── types.ts              # Include JobPayload union + job registry types
│           ├── config.ts
│           ├── logger.ts
│           ├── kafka.ts
│           ├── producer-service.ts
│           ├── result-store.ts
│           └── index.ts
├── docs/
├── package.json
└── pnpm-workspace.yaml
```

- **No root `modules/`** unless you explicitly use it (e.g. as a separate workspace for shared libraries); then add it to `pnpm-workspace.yaml`.

---

## Checklist for adding a new domain module (implemented)

1. Create `apps/api/src/modules/<name>/index.ts` with `{ name, register }` and `export default <name>Module`.
2. Add `<name>` to `MODULE_NAMES` in `apps/api/src/modules/manifest.ts`.
3. Add config for the module in `config.modules` in `packages/shared/src/config.ts`.

## Checklist for adding a new job type (implemented)

1. Add payload type and extend `JobPayload` in `packages/shared/src/types.ts`.
2. Register a builder in `apps/api/src/routes/job-registry.ts` via `registerJobBuilder('your.jobType', ...)`.
3. Add handler in `apps/worker/src/handlers/<name>.ts` and register in `handlers/index.ts` in the `handlers` map.
4. If it’s a stream job, add to `STREAM_JOB_TYPES` in `stream-processor.ts` and handle there (or reuse handler).

---

## Summary

- **Consolidate modules** in `apps/api/src/modules/` and remove or clarify root `modules/`.
- **Convention-based module discovery** (and optional config convention) to avoid editing registry for every new module.
- **Job registry** (map jobType → payload + handler) so new job types are added in one place.
- **Optional:** routes index, deep health from modules, Redis/DB ResultStore, single `app.services` for shared services.

These changes keep your current architecture and make it easier to add modules and job types without touching multiple files.
