/** Centralized config from env */

import type { ModuleConfig } from './types.js';

export interface AppConfig {
  env: 'development' | 'staging' | 'production' | 'test';
  app: { name: string; port: number; host: string };
  kafka: {
    brokers: string[];
    clientId: string;
    consumerGroup: string;
    streamProcessorGroup: string;
    dlqProcessorGroup: string;
    topics: { jobs: string; dlq: string };
    /** Azure Event Hubs (Kafka): set SASL/SSL env vars */
    sasl?: { mechanism: 'plain'; username: string; password: string };
    ssl?: boolean;
  };
  modules: Record<string, ModuleConfig>;
  observability: { logLevel: string; enableTracing: boolean; metricsPort?: number };
}

function getEnv(key: string, defaultValue?: string): string {
  const value = process.env[key] ?? defaultValue;
  if (value === undefined) throw new Error(`Missing required env: ${key}`);
  return value;
}

function getEnvNumber(key: string, defaultValue: number): number {
  const raw = process.env[key];
  if (raw === undefined || raw === '') return defaultValue;
  const n = Number(raw);
  if (Number.isNaN(n)) throw new Error(`Invalid number for env: ${key}`);
  return n;
}

export function loadConfig(): AppConfig {
  const env = (process.env.NODE_ENV ?? 'development') as AppConfig['env'];
  return {
    env,
    app: {
      name: getEnv('APP_NAME', 'MAFGateway-2026'),
      port: getEnvNumber('PORT', 3000),
      host: getEnv('HOST', '0.0.0.0'),
    },
    kafka: {
      brokers: getEnv('KAFKA_BROKERS', 'localhost:9092').split(',').map((b) => b.trim()),
      clientId: getEnv('KAFKA_CLIENT_ID', 'mafgateway-2026'),
      consumerGroup: getEnv('KAFKA_CONSUMER_GROUP', 'mafgateway-workers'),
      streamProcessorGroup: getEnv('KAFKA_STREAM_GROUP', 'mafgateway-stream-processors'),
      dlqProcessorGroup: getEnv('KAFKA_DLQ_GROUP', 'mafgateway-dlq-processors'),
      topics: {
        jobs: getEnv('KAFKA_TOPIC_JOBS', 'mafgateway.jobs'),
        dlq: getEnv('KAFKA_TOPIC_DLQ', 'mafgateway.jobs.dlq'),
      },
      ...(process.env.KAFKA_SASL_PASSWORD
        ? {
            sasl: {
              mechanism: 'plain' as const,
              username: process.env.KAFKA_SASL_USERNAME ?? '$ConnectionString',
              password: process.env.KAFKA_SASL_PASSWORD,
            },
            ssl: true,
          }
        : {}),
    },
    modules: {
      hello: { name: 'hello', enabled: true, basePath: '/api/v1/hello' },
      auth: { name: 'auth', enabled: true, basePath: '/api/v1/auth' },
      permits: { name: 'permits', enabled: true, basePath: '/api/v1/permits' },
      incidents: { name: 'incidents', enabled: true, basePath: '/api/v1/incidents' },
      analytics: { name: 'analytics', enabled: true, basePath: '/api/v1/analytics' },
      notifications: { name: 'notifications', enabled: true, basePath: '/api/v1/notifications' },
    },
    observability: {
      logLevel: getEnv('LOG_LEVEL', env === 'production' ? 'info' : 'debug'),
      enableTracing: process.env.ENABLE_TRACING === 'true',
      metricsPort: getEnvNumber('METRICS_PORT', 9090),
    },
  };
}
