/** Structured logging */

import pino, { type LoggerOptions, type Logger } from 'pino';

export interface LoggerContext {
  requestId?: string;
  module?: string;
  correlationId?: string;
  [key: string]: unknown;
}

const defaultOptions: LoggerOptions = {
  level: process.env.LOG_LEVEL ?? 'info',
  formatters: { level: (label) => ({ level: label }), bindings: () => ({}) },
  timestamp: pino.stdTimeFunctions.isoTime,
  ...(process.env.NODE_ENV !== 'production'
    ? { transport: { target: 'pino-pretty', options: { colorize: true, translateTime: 'SYS:standard' } } }
    : {}),
};

let rootLogger: Logger | null = null;

export function createLogger(options: Partial<LoggerOptions> = {}): Logger {
  return pino({ ...defaultOptions, ...options });
}

export function getLogger(): Logger {
  if (!rootLogger) rootLogger = createLogger();
  return rootLogger;
}

export function setLogger(logger: Logger): void {
  rootLogger = logger;
}

export function childLogger(context: LoggerContext): Logger {
  return getLogger().child(context);
}
