/**
 * Standalone queue consumer – for Stream Processors and DLQ Processor.
 * Consumes from a topic without producer/retry (consumer-only).
 */

import { Kafka } from 'kafkajs';
import type { KafkaClientConfig } from './kafka.js';
import { childLogger } from './logger.js';

const logger = childLogger({ module: 'queue-consumer' });

export interface QueueConsumerConfig extends Pick<KafkaClientConfig, 'brokers' | 'sasl' | 'ssl'> {
  clientId: string;
}

export interface QueueConsumer {
  start(): Promise<void>;
  disconnect(): Promise<void>;
}

/**
 * Create a consumer that reads from a topic and calls onMessage for each record.
 * Used by Stream Processors and DLQ Processor.
 */
export function createQueueConsumer(
  config: QueueConsumerConfig,
  groupId: string,
  topic: string,
  onMessage: (payload: { key: string | null; value: string; headers?: Record<string, string> }) => Promise<void>
): QueueConsumer {
  const kafka = new Kafka({
    clientId: config.clientId,
    brokers: config.brokers,
    ssl: config.ssl ?? false,
    sasl: config.sasl,
    retry: { retries: 3, initialRetryTime: 300, maxRetryTime: 10000 },
  });
  const consumer = kafka.consumer({ groupId });

  return {
    async start(): Promise<void> {
      await consumer.connect();
      await consumer.subscribe({ topic, fromBeginning: false });
      await consumer.run({
        eachMessage: async ({ message }) => {
          const value = message.value?.toString();
          if (!value) return;
          try {
            await onMessage({
              key: message.key?.toString() ?? null,
              value,
              headers: message.headers ? Object.fromEntries(
                Object.entries(message.headers).map(([k, v]) => [k, v?.toString() ?? ''])
              ) : undefined,
            });
          } catch (err) {
            logger.error({ err, topic, key: message.key?.toString() }, 'QueueConsumer: message handler error');
          }
        },
      });
      logger.info({ groupId, topic }, 'QueueConsumer started');
    },
    async disconnect(): Promise<void> {
      await consumer.disconnect();
      logger.info({ groupId, topic }, 'QueueConsumer disconnected');
    },
  };
}
