/**
 * Result Store – sink for Worker Pool and Stream Processors.
 * Persist job results (Database / Cache / Storage). Implementations: in-memory, file, or DB.
 */

import type { JobPayload } from './types.js';
import { childLogger } from './logger.js';

const logger = childLogger({ module: 'result-store' });

export interface JobResult {
  jobId: string;
  jobType: string;
  status: 'completed' | 'failed';
  result?: unknown;
  error?: string;
  completedAt: string;
}

export interface ResultStore {
  save(jobId: string, payload: JobPayload, result: JobResult): Promise<void>;
  get?(jobId: string): Promise<JobResult | null>;
}

/** In-memory store (for dev / single instance). Replace with Redis/DB for production. */
export class InMemoryResultStore implements ResultStore {
  private store = new Map<string, JobResult>();

  async save(_jobId: string, _payload: JobPayload, result: JobResult): Promise<void> {
    this.store.set(result.jobId, result);
    logger.debug({ jobId: result.jobId, status: result.status }, 'ResultStore: saved');
  }

  async get(jobId: string): Promise<JobResult | null> {
    return this.store.get(jobId) ?? null;
  }
}

/** File-based store – writes one JSON file per job under a directory (e.g. for DLQ inspection). */
export class FileResultStore implements ResultStore {
  constructor(private readonly baseDir: string) {}

  async save(_jobId: string, _payload: JobPayload, result: JobResult): Promise<void> {
    const path = await import('node:path');
    const fs = await import('node:fs/promises');
    const file = path.join(this.baseDir, `${result.jobId}.json`);
    await fs.mkdir(this.baseDir, { recursive: true });
    await fs.writeFile(file, JSON.stringify(result, null, 2), 'utf-8');
    logger.debug({ jobId: result.jobId, file }, 'ResultStore: saved to file');
  }
}

export function createDefaultResultStore(): ResultStore {
  return new InMemoryResultStore();
}
