import { forwardRef } from 'react';
import { getImageProps } from 'next/image';
import { processSizes } from './processSizes';
import { processSrcSet } from './processSrcSet';

/**
 * ResponsiveImage component for displaying optimized images with responsive srcSet
 *
 * This component wraps Next.js Image optimization with custom srcSet filtering
 * to ensure only image sizes <= actual image width are included in the srcSet.
 *
 * @example
 * ```tsx
 * <ResponsiveImage
 *   src="/image.jpg"
 *   width={1920}
 *   height={1080}
 *   sizes="100vw"
 *   alt="Description"
 * />
 * ```
 *
 * @example With art direction using Source children
 * ```tsx
 * <ResponsiveImage
 *   src="/image.jpg"
 *   width={576}
 *   height={540}
 *   sizes="lg 50vw, md 640px, sm 100vw"
 * >
 *   <Source media="lg" src="/large.jpg" width={1280} height={540} sizes="100vw" />
 *   <Source media="md" src="/medium.jpg" width={1024} height={540} sizes="100vw" />
 * </ResponsiveImage>
 * ```
 */
export interface ResponsiveImageProps
  extends React.HTMLAttributes<HTMLImageElement> {
  src: string;
  width: number;
  height: number;
  sizes: string;
  alt?: string;
  priority?: boolean;
  quality?: number;
  children?: React.ReactNode;
}

export const ResponsiveImage = forwardRef<HTMLImageElement, ResponsiveImageProps>(
  (
    {
      src,
      width,
      height,
      alt = '',
      sizes,
      quality,
      priority,
      children,
      ...rest
    },
    ref
  ) => {
    if (!src) {
      return null;
    }

    const processedSizes = processSizes(sizes);
    const { props } = getImageProps({
      src,
      width,
      height,
      sizes: processedSizes,
      quality,
      priority,
      alt
    });

    const filteredSrcSet = processSrcSet(props.srcSet, width);

    return (
      <picture>
        {children}
        <img {...props} {...rest} ref={ref} srcSet={filteredSrcSet} alt={alt} />
      </picture>
    );
  }
);

ResponsiveImage.displayName = 'ResponsiveImage';
