import { memo } from 'react';
import { getImageProps } from 'next/image';
import breakpoints, { type BreakpointKey } from './breakpoints';
import { processSizes } from './processSizes';
import { processSrcSet } from './processSrcSet';

export interface SourceProps extends React.HTMLAttributes<HTMLSourceElement> {
  src: string;
  width: number;
  height: number;
  media: string;
  sizes: string;
  quality?: number;
}

/**
 * Source component for art direction with ResponsiveImage
 *
 * Used as a child of ResponsiveImage to provide different image sources
 * for different media queries. The srcSet is automatically filtered to exclude
 * widths greater than the actual image width to exclude redundant image sizes.
 *
 * @example
 * ```tsx
 * <ResponsiveImage src="/default.jpg" width={576} height={540} sizes="100vw">
 *   <Source
 *     media="lg"
 *     src="/large.jpg"
 *     width={1280}
 *     height={540}
 *     sizes="100vw"
 *   />
 *   <Source
 *     media="md"
 *     src="/medium.jpg"
 *     width={1024}
 *     height={540}
 *     sizes="100vw"
 *   />
 * </ResponsiveImage>
 * ```
 *
 * @example With custom media query
 * ```tsx
 * <Source
 *   media="(min-width: 900px)"
 *   src="/custom.jpg"
 *   width={1200}
 *   height={600}
 *   sizes="50vw"
 * />
 * ```
 */
export const Source = memo(
  ({ src, width, height, sizes, media, quality, ...rest }: SourceProps) => {
    if (!src) {
      return null;
    }

    const query = (media in breakpoints ? breakpoints[media as BreakpointKey] : null) || media;
    const processedSizes = processSizes(sizes);
    const { props } = getImageProps({
      width,
      height,
      src,
      sizes: processedSizes,
      quality,
      alt: ''
    });

    const filteredSrcSet = processSrcSet(props.srcSet, width);

    return (
      <source
        media={query}
        srcSet={filteredSrcSet || props.src}
        width={props.width}
        height={props.height}
        sizes={props.sizes}
        {...rest}
      />
    );
  }
);

Source.displayName = 'Source';
