import type { ImageLoader } from 'next/image';

/**
 * Image formats that can be optimized/processed
 * These formats will be optimized, all other formats will be unoptimized
 */
export const processableFormats = ['.jpg', '.jpeg', '.png', '.webp', '.avif'];

/**
 * Checks if an image source should be optimized based on its file extension
 * @param src - The image source path
 * @returns true if the image format is processable (should be optimized), false otherwise
 */
export const shouldOptimizeImage = (src: string): boolean => {
  const pathname = src.split('?')[0];
  return processableFormats.some((format) =>
    pathname.toLowerCase().endsWith(format)
  );
};

/**
 * Custom image loader for Next.js ResponsiveImage component
 * Expects src to be a path starting with /assets/ or /_next/static/media/
 * Only processable image formats (jpg, jpeg, png, webp, avif) are transformed
 * Other formats (svg, gif, etc.) are returned as-is
 */
export const imageLoader: ImageLoader = ({ src, width, quality }) => {
  const pathname = src.split('?')[0];

  const pathMappings: Record<string, string> = {
    '/assets/': '/images/',
    '/_next/static/media/': '/images/static/'
  };

  const basePath = Object.keys(pathMappings).find((path) =>
    pathname.startsWith(path)
  );

  if (!basePath) {
    return src;
  }

  if (!shouldOptimizeImage(src)) {
    return src;
  }

  const q = quality ?? 75;
  return `${pathname.replace(basePath, pathMappings[basePath])}?w=${width}&q=${q}`;
};
