import breakpoints, { type BreakpointKey } from './breakpoints';

// Breakpoint order from largest to smallest for proper sizes attribute ordering
const breakpointOrderMap: Record<BreakpointKey, number> = {
  '2xl': 0,
  xl: 1,
  lg: 2,
  md: 3,
  sm: 4
};

/**
 * Processes the sizes attribute by replacing breakpoint names with media queries
 * and ordering them from largest to smallest breakpoint (required for correct evaluation)
 * Example: "sm 640px, lg 50vw, 85vw" -> "(min-width: 1024px) 50vw, (min-width: 640px) 640px, 85vw"
 */
export function processSizes(sizes?: string): string | undefined {
  if (!sizes) return sizes;

  const parts = sizes.split(',');
  const breakpointEntries: Array<{ order: number; value: string }> = [];
  const defaultEntries: string[] = [];

  // Parse each part
  for (const part of parts) {
    const trimmed = part.trim();
    if (!trimmed) continue;

    // Check if the part starts with a breakpoint name
    const breakpointMatch = trimmed.match(/^(sm|md|lg|xl|2xl)\s+/);
    if (breakpointMatch) {
      const breakpointKey = breakpointMatch[1] as BreakpointKey;

      // Runtime validation: ensure breakpoint exists
      if (breakpointKey in breakpoints) {
        const mediaQuery = breakpoints[breakpointKey];
        const sizeValue = trimmed.substring(breakpointMatch[0].length).trim();
        const processedValue = `${mediaQuery} ${sizeValue}`;

        breakpointEntries.push({
          order: breakpointOrderMap[breakpointKey],
          value: processedValue
        });
      }
    } else {
      // Default entry (no breakpoint) - keep at the end
      defaultEntries.push(trimmed);
    }
  }

  // Sort breakpoint entries by order (smallest order number = largest breakpoint first)
  breakpointEntries.sort((a, b) => a.order - b.order);

  // Combine: breakpoints (largest to smallest) + defaults
  return [...breakpointEntries.map((e) => e.value), ...defaultEntries].join(
    ', '
  );
}
