/**
 * Processes the srcSet string to include widths up to maxWidth, plus one size larger if maxWidth doesn't exactly match any width
 * @param srcSet - The srcSet string from Next.js getImageProps (e.g., "/image.jpg?w=640 640w, /image.jpg?w=1024 1024w")
 * @param maxWidth - The actual width of the image
 * @returns Filtered srcSet string with widths <= maxWidth, plus one size up only if maxWidth doesn't exactly match any width
 * @example
 * // maxWidth exactly matches a width - no next size up included
 * processSrcSet("/img.jpg?w=640 640w, /img.jpg?w=1024 1024w, /img.jpg?w=1200 1200w", 1024)
 * // Returns: "/img.jpg?w=640 640w, /img.jpg?w=1024 1024w"
 *
 * @example
 * // maxWidth doesn't match any width - includes next size up
 * processSrcSet("/img.jpg?w=640 640w, /img.jpg?w=1024 1024w, /img.jpg?w=1200 1200w", 1000)
 * // Returns: "/img.jpg?w=640 640w, /img.jpg?w=1024 1024w"
 *
 * @example
 * // maxWidth smaller than all widths - includes smallest width
 * processSrcSet("/img.jpg?w=640 640w, /img.jpg?w=1024 1024w", 500)
 * // Returns: "/img.jpg?w=640 640w"
 *
 * @example
 * // maxWidth larger than all widths - includes all widths
 * processSrcSet("/img.jpg?w=640 640w, /img.jpg?w=1024 1024w", 2000)
 * // Returns: "/img.jpg?w=640 640w, /img.jpg?w=1024 1024w"
 *
 * @example
 * // Density descriptors are always included
 * processSrcSet("/img.jpg?w=640 640w, /img.jpg?w=1024 1024w, /img.jpg?w=1280 2x", 1000)
 * // Returns: "/img.jpg?w=640 640w, /img.jpg?w=1024 1024w, /img.jpg?w=1280 2x"
 */
export function processSrcSet(
  srcSet: string | undefined,
  maxWidth: number
): string | undefined {
  if (!srcSet || typeof srcSet !== 'string') {
    return srcSet;
  }

  // Split srcSet by comma to get individual entries
  const entries = srcSet
    .split(',')
    .map((entry) => entry.trim())
    .filter(Boolean);

  // First pass: collect all entries with their widths and find the next size up
  type EntryWithWidth = {
    entry: string;
    width: number | null;
  };

  const entriesWithWidths: EntryWithWidth[] = entries.map((entry) => {
    // Extract width descriptor (e.g., "640w" from "/image.jpg?w=640 640w")
    const widthMatch = entry.match(/\s+(\d+)w$/);
    const width = widthMatch ? parseInt(widthMatch[1], 10) : null;
    return { entry, width };
  });

  // Check if maxWidth exactly matches any width in the srcSet
  const hasExactMatch = entriesWithWidths.some(
    (e) => e.width !== null && e.width === maxWidth
  );

  // Find the smallest width that is greater than maxWidth (one size up)
  // Only include next size up if maxWidth doesn't exactly match any width
  const widthsGreaterThanMax = entriesWithWidths
    .filter((e) => e.width !== null && e.width > maxWidth)
    .map((e) => e.width!);
  const nextSizeUp =
    !hasExactMatch && widthsGreaterThanMax.length > 0
      ? Math.min(...widthsGreaterThanMax)
      : null;

  // Filter entries: include widths <= maxWidth, plus the next size up if it exists and maxWidth doesn't match exactly
  const filteredEntries = entriesWithWidths
    .filter(({ width }) => {
      // Include density descriptors (they don't have width, so width is null)
      if (width === null) {
        return true;
      }
      // Include widths <= maxWidth
      if (width <= maxWidth) {
        return true;
      }
      // Include the next size up only if maxWidth doesn't exactly match any width
      if (nextSizeUp !== null && width === nextSizeUp) {
        return true;
      }
      return false;
    })
    .map(({ entry }) => entry);

  // Return the filtered srcSet or undefined if empty
  return filteredEntries.length > 0 ? filteredEntries.join(', ') : undefined;
}
