import React from 'react';
import { Meta } from '@storybook/nextjs';

type ColorsItem = {
  color: string;
  className?: string;
  textClassName?: string;
};

type SwatchColors = {
  title: string;
  items: ColorsItem[];
};

type SwatchSetProps = {
  colors: SwatchColors[];
};

const SemanticSwatch = ({
  color,
  className,
  textClassName
}: {
  color: string;
  className?: string;
  textClassName?: string;
}) => {
  return (
    <div
      className={`${className} ${textClassName} text-sm flex flex-col items-center justify-center w-24 h-24 m-1 rounded shadow-lg`}>
      <span>{color}</span>
    </div>
  );
};

const SwatchSet = ({ colors }: SwatchSetProps) => (
  <div className="flex flex-row flex-wrap items-center justify-center w-full h-full py-2">
    {colors.map(({ title, items }) => (
      <div key={title} className="flex flex-col items-start w-full h-full">
        <h2 className="text-xl font-bold text-foreground">{title}</h2>
        <div className="flex flex-row flex-wrap items-center justify-start w-full h-full py-4">
          {items.map((c, index) => (
            <SemanticSwatch
              key={`${c.color}-${index}`}
              className={c.className}
              color={c.color}
              textClassName={c.textClassName}
            />
          ))}
        </div>
      </div>
    ))}
  </div>
);

// Background color key -> Text color value mapping
/** we need it for tailwind to detect the color */
const COLOR_CLASS_MAP = {
  layout: {
    'bg-background': 'text-foreground',
    'bg-foreground': 'text-background',
    'bg-divider': 'text-foreground',
    'bg-focus': 'text-primary-foreground'
  },
  base: {
    'bg-default': 'text-default-foreground',
    'bg-primary': 'text-primary-foreground',
    'bg-secondary': 'text-secondary-foreground',
    'bg-danger': 'text-danger-foreground',
    'bg-warning': 'text-warning-foreground',
    'bg-success': 'text-success-foreground'
  },
  default: {
    'bg-default-50': 'text-default-900',
    'bg-default-100': 'text-default-900',
    'bg-default-200': 'text-default-800',
    'bg-default-300': 'text-default-800',
    'bg-default-400': 'text-default-800',
    'bg-default-500': 'text-default-foreground',
    'bg-default-600': 'text-default-50',
    'bg-default-700': 'text-default-100',
    'bg-default-800': 'text-default-100',
    'bg-default-900': 'text-default-100'
  },
  primary: {
    'bg-primary-50': 'text-primary-900',
    'bg-primary-100': 'text-primary-900',
    'bg-primary-200': 'text-primary-800',
    'bg-primary-300': 'text-primary-800',
    'bg-primary-400': 'text-primary-800',
    'bg-primary-500': 'text-primary-foreground',
    'bg-primary-600': 'text-primary-50',
    'bg-primary-700': 'text-primary-100',
    'bg-primary-800': 'text-primary-100',
    'bg-primary-900': 'text-primary-100'
  },
  secondary: {
    'bg-secondary-50': 'text-secondary-900',
    'bg-secondary-100': 'text-secondary-700',
    'bg-secondary-200': 'text-secondary-700',
    'bg-secondary-300': 'text-secondary-700',
    'bg-secondary-400': 'text-secondary-700',
    'bg-secondary-500': 'text-secondary-50',
    'bg-secondary-600': 'text-secondary-50',
    'bg-secondary-700': 'text-secondary-100',
    'bg-secondary-800': 'text-secondary-100',
    'bg-secondary-900': 'text-secondary-100'
  }
};

const generateColorScale = (colorName: string): ColorsItem[] => {
  const colorMap = COLOR_CLASS_MAP[colorName as keyof typeof COLOR_CLASS_MAP];
  if (!colorMap) return [];

  return Object.keys(colorMap).map((bgClassName) => {
    // Extract color name from bg class (e.g., 'bg-default-50' -> 'default-50')
    const colorDisplayName = bgClassName.replace('bg-', '');

    return {
      color: colorDisplayName,
      className: bgClassName,
      textClassName: colorMap[bgClassName as keyof typeof colorMap]
    };
  });
};

const generateColorSections = (): SwatchColors[] => {
  const colorNames = Object.keys(COLOR_CLASS_MAP);

  return colorNames.map((colorName) => ({
    title: colorName.charAt(0).toUpperCase() + colorName.slice(1),
    items: generateColorScale(colorName)
  }));
};

export default {
  title: 'Foundations/01-SemanticColors',
  component: SwatchSet
} as Meta<typeof SwatchSet>;

export const SemanticColors = {
  args: {
    colors: generateColorSections()
  }
};
