/**
 * Processes the srcSet string to exclude width descriptors greater than the actual image width
 * @param srcSet - The srcSet string from Next.js getImageProps (e.g., "/image.jpg?w=640 640w, /image.jpg?w=1024 1024w")
 * @param maxWidth - The actual width of the image
 * @returns Filtered srcSet string with only widths <= maxWidth
 */
export function processSrcSet(srcSet: string | undefined, maxWidth: number): string | undefined {
  if (!srcSet || typeof srcSet !== 'string') {
    return srcSet;
  }

  // Split srcSet by comma to get individual entries
  const entries = srcSet.split(',').map(entry => entry.trim()).filter(Boolean);

  // Filter and process each entry
  const filteredEntries = entries
    .map(entry => {
      // Extract width descriptor (e.g., "640w" from "/image.jpg?w=640 640w")
      // Also handle density descriptors (e.g., "2x") - these should be included as-is
      const widthMatch = entry.match(/\s+(\d+)w$/);
      const densityMatch = entry.match(/\s+(\d+(?:\.\d+)?)x$/);

      // If it's a density descriptor (e.g., "2x"), include it as-is
      if (densityMatch) {
        return entry;
      }

      // If it's a width descriptor, check against maxWidth
      if (widthMatch) {
        const width = parseInt(widthMatch[1], 10);

        // Include entry only if width is less than or equal to maxWidth
        if (width <= maxWidth) {
          return entry;
        }

        return null;
      }

      // If no descriptor found, include the entry as-is (fallback)
      return entry;
    })
    .filter((entry): entry is string => entry !== null);

  // Return the filtered srcSet or undefined if empty
  return filteredEntries.length > 0 ? filteredEntries.join(', ') : undefined;
}

