'use client';

import React, { useCallback, useEffect, useRef, useState } from 'react';
import breakpoints, { type BreakpointKey } from './breakpoints';

export type IResponsiveVideoProps = React.VideoHTMLAttributes<HTMLVideoElement> & {
  src?: string;
  /**
   * First match in the array will be selected.
   * media should be a valid media query or a key from breakpoints object.
   */
  srcSet?: {
    src: string;
    media: string;
  }[];
  /** auto plays video source */
  autoPlay?: boolean;
  /**
   * If autoplay fails or src is not available and if there is a fallback,
   * display this fallback element
   */
  fallback?: React.ReactNode;
};

/**
 * Resolves media query string from breakpoint key or returns the media query as-is.
 * If media is a breakpoint key (e.g., 'md'), it returns the corresponding media query.
 * Otherwise, it assumes media is already a valid media query string.
 */
const getMediaQuery = (media: string) =>
  (media in breakpoints ? breakpoints[media as BreakpointKey] : null) || media;

export const ResponsiveVideo = React.forwardRef<
  HTMLVideoElement,
  IResponsiveVideoProps
>(({ src, srcSet, autoPlay, fallback, onError, ...rest }, ref) => {
  const [isAutoPlayFailed, setIsAutoPlayFailed] = useState(false);
  const [hasError, setHasError] = useState(false);
  const [currentSrc, setCurrentSrc] = useState<string | undefined>(src);
  const videoRef = useRef<HTMLVideoElement>(null);

  /**
   * Ref callback that handles both function refs and object refs.
   * We maintain an internal videoRef to access the video element for autoplay logic,
   * while also forwarding the ref to the parent component.
   */
  const setRefs = useCallback(
    (node: HTMLVideoElement | null) => {
      videoRef.current = node;
      if (!ref) return;
      if (typeof ref === 'function') {
        ref(node);
      } else {
        (ref as React.RefObject<HTMLVideoElement | null>).current = node;
      }
    },
    [ref]
  );

  useEffect(() => {
    // SSR safety: window is not available during server-side rendering
    if (typeof window === 'undefined') {
      setCurrentSrc(src);
      return undefined;
    }

    // If no srcSet provided, use default src
    if (!srcSet?.length) {
      setCurrentSrc(src);
      return undefined;
    }

    // Create MediaQueryList listeners for each media query in srcSet.
    // Using MediaQueryList listeners instead of window resize events is more performant
    // as it only fires when the specific media query state changes, not on every resize.
    const mediaQueries = srcSet.map(({ media, src: source }) => {
      const query = getMediaQuery(media);
      const mediaQueryList = window.matchMedia(query);

      const handleChange = (event: MediaQueryListEvent) => {
        // Only update source when the media query becomes active (matches = true)
        if (event.matches) {
          setCurrentSrc(source);
        }
      };

      mediaQueryList.addEventListener('change', handleChange);

      return { mediaQueryList, handleChange };
    });

    // On mount, find the first matching media query to set initial source.
    // This ensures the correct video source is selected immediately, not waiting for a resize event.
    const selectSource = () => {
      const matchedSet = srcSet.find(({ media }) => {
        const query = getMediaQuery(media);
        return window.matchMedia(query).matches;
      });
      setCurrentSrc(matchedSet?.src || src);
    };

    selectSource();

    // Cleanup: remove all event listeners to prevent memory leaks
    return () =>
      mediaQueries.forEach(({ mediaQueryList, handleChange }) =>
        mediaQueryList.removeEventListener('change', handleChange)
      );
  }, [src, srcSet]);

  useEffect(() => {
    if (!autoPlay) return;
    const node = videoRef.current;
    if (!node) return;

    // Respect user's reduced motion preference for accessibility.
    // If user prefers reduced motion, we skip autoplay and show fallback if available.
    if (typeof window !== 'undefined') {
      const prefersReducedMotion = window.matchMedia(
        '(prefers-reduced-motion: reduce)'
      );

      if (prefersReducedMotion.matches) {
        setIsAutoPlayFailed(true);
        return;
      }
    }

    // Attempt to play the video. AbortError occurs when the video source changes
    // while a play() promise is pending (e.g., during responsive source switching).
    // We ignore AbortError as it's expected behavior, not a real failure.
    node.play().catch((error: unknown) => {
      if (error instanceof DOMException && error.name === 'AbortError') {
        return;
      }
      setIsAutoPlayFailed(true);
    });
  }, [autoPlay, currentSrc]);

  const handleError = (event: React.SyntheticEvent<HTMLVideoElement, Event>) => {
    setHasError(true);
    // Forward the error event to parent component's onError handler if provided
    onError?.(event);
  };

  // Show fallback if: autoplay failed, video load error occurred, or no source available.
  // Fallback is only shown if it's provided as a prop.
  const shouldShowFallback = (isAutoPlayFailed || hasError || !currentSrc) && fallback;

  if (shouldShowFallback) {
    return fallback;
  }

  if (!currentSrc) {
    return null;
  }

  return (
    <video
      ref={setRefs}
      src={currentSrc}
      onError={handleError}
      {...rest}
    />
  );
});

ResponsiveVideo.displayName = 'ResponsiveVideo';
