import express from 'express'
import cors from 'cors'
import dotenv from 'dotenv'
import wheelRoutes from './routes/wheelRoutes.js'
import entryRoutes from './routes/entryRoutes.js'
import spinRoutes from './routes/spinRoutes.js'
import resultRoutes from './routes/resultRoutes.js'

dotenv.config()

const app = express()
const PORT = process.env.PORT || 5000

// CORS configuration - allow dynamic frontend URL
const corsOptions = {
  origin: function (origin: string | undefined, callback: (err: Error | null, allow?: boolean) => void) {
    // Allow requests with no origin (like mobile apps or curl requests)
    if (!origin) return callback(null, true)
    
    // Get allowed origins from environment variable or use default
    const allowedOrigins = process.env.ALLOWED_ORIGINS 
      ? process.env.ALLOWED_ORIGINS.split(',').map(o => o.trim())
      : ['http://localhost:3000', 'http://localhost:5173']
    
    // Check if origin is allowed
    if (allowedOrigins.includes(origin) || allowedOrigins.includes('*')) {
      callback(null, true)
    } else {
      // For production, you might want to be more strict
      // For development, allow all origins
      if (process.env.NODE_ENV === 'development') {
        callback(null, true)
      } else {
        callback(new Error('Not allowed by CORS'))
      }
    }
  },
  credentials: true,
  optionsSuccessStatus: 200
}

// Middleware
app.use(cors(corsOptions))
app.use(express.json())

// Routes
app.use('/api/wheels', wheelRoutes)
app.use('/api/entries', entryRoutes)
app.use('/api/spin', spinRoutes)
app.use('/api/results', resultRoutes)

// Health check
app.get('/api/health', (req, res) => {
  res.json({ status: 'ok', message: 'Spin The Wheel API is running' })
})

app.listen(PORT, () => {
  // Server started successfully
})

