export interface PhysicsState {
  angle: number
  velocity: number
  deceleration: number
}

export class WheelPhysics {
  private angle: number = 0 // Current angle in radians
  private angleDelta: number = 0 // Current angular velocity
  private maxSpeed: number = Math.PI / 16
  private isSpinning: boolean = false
  private spinStartTime: number = 0
  private upTime: number = 1000 // Spin up time in ms
  private downTime: number = 5000 // Spin down time in ms

  /**
   * Start spinning the wheel with natural physics
   * Uses sine-based acceleration/deceleration like the reference
   */
  startSpin(spinTime: number = 10) {
    this.isSpinning = true
    this.spinStartTime = performance.now()
    
    // Randomly vary the spin intensity (like reference)
    const array = new Uint32Array(1)
    crypto.getRandomValues(array)
    const randomValue = array[0] / (0xFFFFFFFF + 1)
    this.maxSpeed = Math.PI / (16 + randomValue * 8) // Vary between PI/16 and PI/8
    
    // Convert spinTime to milliseconds for upTime/downTime
    this.upTime = Math.min(1000, spinTime * 1000 * 0.1) // 10% of time for spin up
    this.downTime = spinTime * 1000 - this.upTime // Rest for spin down
  }

  /**
   * Update physics based on elapsed time
   * Returns current angle and whether spinning is complete
   */
  update(currentTime: number): { angle: number; isComplete: boolean } {
    if (!this.isSpinning) {
      return { angle: this.angle, isComplete: true }
    }

    const duration = currentTime - this.spinStartTime
    let progress = 0
    let finished = false

    // Spin up phase (acceleration)
    if (duration < this.upTime) {
      progress = duration / this.upTime
      // Sine curve for smooth acceleration: sin(progress * PI/2)
      this.angleDelta = this.maxSpeed * Math.sin((progress * Math.PI) / 2)
    } 
    // Spin down phase (deceleration)
    else {
      progress = (duration - this.upTime) / this.downTime
      // Sine curve for smooth deceleration: sin(progress * PI/2 + PI/2)
      this.angleDelta = this.maxSpeed * Math.sin((progress * Math.PI) / 2 + Math.PI / 2)
      if (progress >= 1) {
        finished = true
      }
    }

    // Update angle
    this.angle += this.angleDelta

    // Keep angle in range [0, 2π)
    const PI2 = Math.PI * 2
    while (this.angle >= PI2) {
      this.angle -= PI2
    }
    while (this.angle < 0) {
      this.angle += PI2
    }

    if (finished) {
      this.isSpinning = false
      this.angleDelta = 0
      return { angle: this.angle, isComplete: true }
    }

    return { angle: this.angle, isComplete: false }
  }

  /**
   * Get current angle in radians
   */
  getAngle(): number {
    return this.angle
  }

  /**
   * Get current angle in degrees
   */
  getAngleDegrees(): number {
    return (this.angle * 180) / Math.PI
  }

  isCurrentlySpinning(): boolean {
    return this.isSpinning
  }

  stop() {
    this.isSpinning = false
    this.angleDelta = 0
  }
}

// Cryptographically secure random selection
export function getSecureRandomEntry<T>(entries: T[]): T {
  if (entries.length === 0) {
    throw new Error('Cannot select from empty array')
  }

  const array = new Uint32Array(1)
  crypto.getRandomValues(array)
  const randomValue = array[0] / (0xFFFFFFFF + 1)
  const index = Math.floor(randomValue * entries.length)
  
  return entries[index]
}

/**
 * Calculates the winner index based on rotation angle.
 * 
 * Logic:
 * 1. Pointer is fixed at 0 radians (3 o'clock / right side).
 * 2. Wheel drawing starts with segment 0 at -PI/2 radians (12 o'clock / top).
 * 3. Wheel is rotated clockwise by 'angle'.
 * 4. The local angle on the wheel that aligns with the global pointer (0) is:
 *    localAngle + (angle - PI/2) = 0  =>  localAngle = PI/2 - angle
 * 
 * @param angle - Current wheel angle in radians
 * @param totalSegments - Total number of segments
 * @returns The index of the segment being pointed to (0-based)
 */
export function getSegmentIndexFromAngle(angle: number, totalSegments: number): number {
  if (totalSegments === 0) return 0;
  
  // Normalize angle to [0, 2PI)
  const normalizedAngle = ((angle % (Math.PI * 2)) + (Math.PI * 2)) % (Math.PI * 2);
  
  // Calculate which local wheel angle is at the global 0 position
  let localAngleAtPointer = (Math.PI / 2) - normalizedAngle;
  
  // Normalize localAngleAtPointer to [0, 2PI)
  localAngleAtPointer = ((localAngleAtPointer % (Math.PI * 2)) + (Math.PI * 2)) % (Math.PI * 2);
  
  const arcSize = (Math.PI * 2) / totalSegments;
  const index = Math.floor(localAngleAtPointer / arcSize);
  
  return index % totalSegments;
}
