import { create } from 'zustand'
import { Entry, WheelSettings, Result } from '../types'

interface WheelState {
  entries: Entry[]
  results: Result[]
  settings: WheelSettings
  isSpinning: boolean
  winner: Entry | null
  lastWinner: Entry | null // Alias for winner to match professional version
  currentWheelId: string | null
  addEntry: (text: string) => void
  bulkAddEntries: (texts: string[]) => void
  removeEntry: (id: string) => void
  updateEntry: (id: string, updates: Partial<Entry>) => void
  shuffleEntries: () => void
  sortEntries: () => void
  clearEntries: () => void
  addResult: (entry: Entry) => void
  removeResult: (id: string) => void
  clearResults: () => void
  updateSettings: (updates: Partial<WheelSettings>) => void
  setSpinning: (spinning: boolean) => void
  setWinner: (winner: Entry | null) => void
  setLastWinner: (winner: Entry | null) => void // Alias for setWinner
  setWheelId: (id: string | null) => void
  loadEntries: (entries: Entry[]) => void
  loadResults: (results: Result[]) => void
  loadSettings: (settings: WheelSettings) => void
}

const defaultSettings: WheelSettings = {
  duringSpin: {
    sound: 'ticking',
    volume: 50,
    displayDuplicates: true,
    spinSlowly: false,
    showTitle: true,
    spinTime: 10,
    maxVisibleNames: 1000,
  },
  afterSpin: {
    sound: 'applause',
    volume: 50,
    animateWinningEntry: false,
    launchConfetti: true,
    autoRemoveWinner: null,
    displayPopup: true,
    popupMessage: 'We have a winner!',
    displayRemoveButton: true,
    playClickSoundOnRemove: false,
  },
  appearance: {
    colorScheme: 'one-color',
    theme: 'default',
    colors: ['#DB4437', '#4285F4', '#0F9D58', '#F4B400', '#DB4437', '#4285F4', '#0F9D58', '#F4B400'],
    centerImageSize: 'S',
    displayGradient: true,
    contours: false,
    wheelShadow: true,
    pointerChangesColor: true,
  },
}

export const useWheelStore = create<WheelState>((set) => ({
  entries: [], // Start with empty entries - will load from DB
  results: [], // Start with empty results - will load from DB
  settings: defaultSettings,
  isSpinning: false,
  winner: null,
  lastWinner: null,
  currentWheelId: null,

  addEntry: (text: string) =>
    set((state) => ({
      entries: [
        ...state.entries,
        { id: Date.now().toString() + Math.random().toString(36).substr(2, 9), text: text.trim() },
      ],
    })),

  bulkAddEntries: (texts: string[]) =>
    set((state) => {
      const newEntries = texts.map((text) => ({
        id: Date.now().toString() + Math.random().toString(36).substr(2, 9) + Math.random().toString(36).substr(2, 9),
        text: text.trim(),
      }))
      return {
        entries: [...state.entries, ...newEntries],
      }
    }),

  removeEntry: (id: string) =>
    set((state) => ({
      entries: state.entries.filter((e) => e.id !== id),
    })),

  updateEntry: (id: string, updates: Partial<Entry>) =>
    set((state) => ({
      entries: state.entries.map((e) =>
        e.id === id ? { ...e, ...updates } : e
      ),
    })),

  shuffleEntries: () =>
    set((state) => {
      const shuffled = [...state.entries]
      for (let i = shuffled.length - 1; i > 0; i--) {
        const j = Math.floor(Math.random() * (i + 1))
        ;[shuffled[i], shuffled[j]] = [shuffled[j], shuffled[i]]
      }
      return { entries: shuffled }
    }),

  sortEntries: () =>
    set((state) => ({
      entries: [...state.entries].sort((a, b) =>
        a.text.localeCompare(b.text)
      ),
    })),

  clearEntries: () => set({ entries: [] }),

  addResult: (entry: Entry) =>
    set((state) => ({
      results: [
        {
          id: Date.now().toString(),
          entryId: entry.id,
          entryText: entry.text,
          spunAt: new Date(),
        },
        ...state.results,
      ],
    })),

  removeResult: (id: string) =>
    set((state) => ({
      results: state.results.filter((r) => r.id !== id),
    })),

  clearResults: () => set({ results: [] }),

  updateSettings: (updates: Partial<WheelSettings>) =>
    set((state) => ({
      settings: { ...state.settings, ...updates },
    })),

  setSpinning: (spinning: boolean) => set({ isSpinning: spinning }),

  setWinner: (winner: Entry | null) => set({ winner, lastWinner: winner }),

  setLastWinner: (winner: Entry | null) => set({ winner, lastWinner: winner }),

  setWheelId: (id: string | null) => set({ currentWheelId: id }),

  loadEntries: (entries: Entry[]) => set({ entries }),

  loadResults: (results: Result[]) => set({ results }),

  loadSettings: (settings: WheelSettings) => set({ settings }),
}))

