import { WheelSettings } from '../../types'
import './settings.css'

interface AppearanceSettingsProps {
  settings: WheelSettings['appearance']
  onUpdate: (updates: Partial<WheelSettings['appearance']>) => void
}

export default function AppearanceSettings({
  settings,
  onUpdate,
}: AppearanceSettingsProps) {
  const defaultColors = [
    '#FF6B6B',
    '#4ECDC4',
    '#45B7D1',
    '#FFA07A',
    '#98D8C8',
    '#F7DC6F',
    '#BB8FCE',
  ]

  const toggleColor = (color: string) => {
    const newColors = settings.colors.includes(color)
      ? settings.colors.filter((c) => c !== color)
      : [...settings.colors, color]
    onUpdate({ colors: newColors })
  }

  return (
    <div className="settings-panel">
      <div className="setting-group">
        <div className="color-scheme-selector">
          <label className="color-scheme-option">
            <input
              type="radio"
              name="colorScheme"
              value="one-color"
              checked={settings.colorScheme === 'one-color'}
              onChange={(e) =>
                onUpdate({ colorScheme: e.target.value as 'one-color' | 'background-image' })
              }
            />
            <span>One color per section</span>
          </label>
          <label className="color-scheme-option">
            <input
              type="radio"
              name="colorScheme"
              value="background-image"
              checked={settings.colorScheme === 'background-image'}
              onChange={(e) =>
                onUpdate({ colorScheme: e.target.value as 'one-color' | 'background-image' })
              }
            />
            <span>Wheel background image</span>
          </label>
        </div>
      </div>

      <hr className="setting-separator" />

      <div className="setting-group">
        <button className="btn-theme">Apply a theme</button>
        <div className="setting-label-with-help">
          <label className="setting-label">Customize colors</label>
          <button className="help-btn">?</button>
        </div>
        <div className="color-palette">
          {defaultColors.map((color, index) => (
            <label key={index} className="color-swatch">
              <input
                type="checkbox"
                checked={settings.colors.includes(color)}
                onChange={() => toggleColor(color)}
              />
              <span
                className="color-box"
                style={{ backgroundColor: color }}
              ></span>
            </label>
          ))}
        </div>
      </div>

      <hr className="setting-separator" />

      <div className="setting-group">
        <label className="setting-label">Image at the center of the wheel</label>
        <button className="btn-image">Add image</button>
        <div className="setting-group-inline">
          <label className="setting-label">Image size</label>
          <select
            value={settings.centerImageSize}
            onChange={(e) =>
              onUpdate({ centerImageSize: e.target.value as 'S' | 'M' | 'L' })
            }
            className="setting-select"
          >
            <option value="S">S</option>
            <option value="M">M</option>
            <option value="L">L</option>
          </select>
        </div>
      </div>

      <hr className="setting-separator" />

      <div className="setting-group">
        <label className="setting-checkbox">
          <input
            type="checkbox"
            checked={settings.displayGradient}
            onChange={(e) => onUpdate({ displayGradient: e.target.checked })}
          />
          <span>Display a color gradient on the page</span>
        </label>
        <label className="setting-checkbox">
          <input
            type="checkbox"
            checked={settings.contours}
            onChange={(e) => onUpdate({ contours: e.target.checked })}
          />
          <span>Contours</span>
        </label>
        <label className="setting-checkbox">
          <input
            type="checkbox"
            checked={settings.wheelShadow}
            onChange={(e) => onUpdate({ wheelShadow: e.target.checked })}
          />
          <span>Wheel shadow</span>
        </label>
        <label className="setting-checkbox">
          <input
            type="checkbox"
            checked={settings.pointerChangesColor}
            onChange={(e) =>
              onUpdate({ pointerChangesColor: e.target.checked })
            }
          />
          <span>Pointer changes color</span>
        </label>
      </div>
    </div>
  )
}

