import { Request, Response } from 'express'
import prisma from '../services/prisma.js'
import { getSecureRandomEntry, getWeightedRandomEntry } from '../services/randomService.js'

export const performSpin = async (req: Request, res: Response) => {
  try {
    const { wheelId, entries: entriesArray } = req.body

    let entries: any[] = []

    if (wheelId) {
      // Get all entries for the wheel from database
      entries = await prisma.entry.findMany({
        where: { wheelId },
        orderBy: { order: 'asc' },
      })
    } else if (entriesArray && Array.isArray(entriesArray) && entriesArray.length > 0) {
      // Use provided entries array (for large datasets without saved wheel)
      entries = entriesArray.map((e: any, index: number) => ({
        id: e.id || `temp-${index}`,
        text: e.text || e,
        weight: e.weight,
        color: e.color,
      }))
    } else {
      return res.status(400).json({ error: 'Either wheelId or entries array is required' })
    }

    if (entries.length === 0) {
      return res.status(400).json({ error: 'No entries found' })
    }

    // Check if any entries have weights (advanced mode)
    const hasWeights = entries.some((e) => e.weight !== null && e.weight !== undefined)

    // Select random entry
    const selectedEntry = hasWeights
      ? getWeightedRandomEntry(entries)
      : getSecureRandomEntry(entries)

    // Save result
    const result = await prisma.result.create({
      data: {
        wheelId,
        entryId: selectedEntry.id,
      },
      include: {
        entry: true,
      },
    })

    res.json({
      winner: {
        id: selectedEntry.id,
        text: selectedEntry.text,
        imageUrl: selectedEntry.imageUrl,
        color: selectedEntry.color,
      },
      result: {
        id: result.id,
        spunAt: result.spunAt,
      },
    })
  } catch (error: any) {
    res.status(500).json({ error: error.message })
  }
}

