import { Entry } from '../types/index.js'

/**
 * Process and validate large entry lists (up to 10,000 entries)
 * Optimizes entries for wheel rendering
 */
export class EntryProcessor {
  /**
   * Validate entry list
   */
  static validateEntries(entries: string[]): { valid: boolean; errors: string[] } {
    const errors: string[] = []

    if (entries.length === 0) {
      errors.push('At least one entry is required')
    }

    if (entries.length > 10000) {
      errors.push('Maximum 10,000 entries allowed')
    }

    // Check for empty entries
    const emptyEntries = entries.filter((e) => !e.trim())
    if (emptyEntries.length > 0) {
      errors.push(`${emptyEntries.length} empty entries found`)
    }

    // Check for extremely long entries
    const longEntries = entries.filter((e) => e.length > 200)
    if (longEntries.length > 0) {
      errors.push(`${longEntries.length} entries exceed 200 characters`)
    }

    return {
      valid: errors.length === 0,
      errors,
    }
  }

  /**
   * Process entries for wheel display
   * Returns optimized subset for frontend rendering
   */
  static processEntries(
    entries: Entry[],
    maxVisible: number
  ): {
    visible: Entry[]
    total: number
    hasMore: boolean
  } {
    const visible = entries.slice(0, maxVisible)
    const total = entries.length
    const hasMore = total > maxVisible

    return {
      visible,
      total,
      hasMore,
    }
  }

  /**
   * Batch process entries for large datasets
   */
  static batchProcessEntries(
    entries: Entry[],
    batchSize: number = 1000
  ): Entry[][] {
    const batches: Entry[][] = []
    for (let i = 0; i < entries.length; i += batchSize) {
      batches.push(entries.slice(i, i + batchSize))
    }
    return batches
  }
}

