import { Request, Response } from 'express'
import prisma from '../services/prisma.js'
import { EntryProcessor } from '../services/entryProcessor.js'

export const validateEntries = async (req: Request, res: Response) => {
  try {
    const { entries } = req.body

    if (!Array.isArray(entries)) {
      return res.status(400).json({ error: 'Entries must be an array' })
    }

    const validation = EntryProcessor.validateEntries(entries)
    res.json(validation)
  } catch (error: any) {
    res.status(500).json({ error: error.message })
  }
}

export const processEntries = async (req: Request, res: Response) => {
  try {
    const { entries, maxVisible } = req.body

    if (!Array.isArray(entries)) {
      return res.status(400).json({ error: 'Entries must be an array' })
    }

    const processed = EntryProcessor.processEntries(entries, maxVisible || 1000)
    res.json(processed)
  } catch (error: any) {
    res.status(500).json({ error: error.message })
  }
}

export const getEntries = async (req: Request, res: Response) => {
  try {
    const { wheelId } = req.params

    const entries = await prisma.entry.findMany({
      where: { wheelId },
      orderBy: { order: 'asc' },
    })

    res.json(entries)
  } catch (error: any) {
    res.status(500).json({ error: error.message })
  }
}

export const addEntries = async (req: Request, res: Response) => {
  try {
    const { wheelId } = req.params
    const { entries } = req.body

    if (!Array.isArray(entries)) {
      return res.status(400).json({ error: 'Entries must be an array' })
    }

    // Get current max order
    const maxOrder = await prisma.entry.aggregate({
      where: { wheelId },
      _max: { order: true },
    })

    const startOrder = (maxOrder._max.order ?? -1) + 1

    const created = await prisma.entry.createMany({
      data: entries.map((entry: any, index: number) => ({
        wheelId,
        text: entry.text || entry,
        imageUrl: entry.imageUrl,
        weight: entry.weight,
        color: entry.color,
        order: startOrder + index,
      })),
    })

    res.json({ count: created.count })
  } catch (error: any) {
    res.status(500).json({ error: error.message })
  }
}

export const removeEntry = async (req: Request, res: Response) => {
  try {
    const { wheelId, entryId } = req.params

    await prisma.entry.delete({
      where: { id: entryId },
    })

    res.json({ message: 'Entry removed successfully' })
  } catch (error: any) {
    res.status(500).json({ error: error.message })
  }
}

export const clearAllEntries = async (req: Request, res: Response) => {
  try {
    const { wheelId } = req.params

    await prisma.entry.deleteMany({
      where: { wheelId },
    })

    res.json({ message: 'All entries cleared successfully' })
  } catch (error: any) {
    res.status(500).json({ error: error.message })
  }
}

