import { useState, useEffect } from 'react'
import WheelCanvas from './components/Wheel/WheelCanvas'
import EntryManager from './components/EntryManager/EntryManager'
import CustomizationModal from './components/Customization/CustomizationModal'
import { useWheelStore } from './store/wheelStore'
import { wheelApi, entryApi, resultApi } from './services/api'
import { Settings, Trophy } from 'lucide-react'
import './App.css'

function App() {
  const [showCustomize, setShowCustomize] = useState(false)
  const { 
    settings, 
    lastWinner, 
    setLastWinner, 
    removeEntry,
    currentWheelId,
    setWheelId,
    loadEntries,
    loadResults,
    loadSettings,
  } = useWheelStore()

  // Load wheel data from database when wheelId changes or on mount
  useEffect(() => {
    const loadWheelData = async () => {
      let wheelId = currentWheelId

      // If no wheelId, try to get from localStorage or load most recent wheel
      if (!wheelId) {
        // Try to get from localStorage first
        const savedWheelId = localStorage.getItem('spin-the-wheel_currentWheelId')
        if (savedWheelId) {
          wheelId = savedWheelId
          setWheelId(wheelId)
        } else {
          // Try to get the most recent wheel from database
          try {
            const recentResponse = await wheelApi.getRecent()
            const latestWheel = recentResponse.data
            if (latestWheel && latestWheel.id) {
              wheelId = latestWheel.id
              setWheelId(wheelId)
              localStorage.setItem('spin-the-wheel_currentWheelId', wheelId)
            } else {
              // No wheels exist, start with empty state
              loadEntries([])
              loadResults([])
              return
            }
          } catch (error: any) {
            // If 404, no wheels exist yet - that's fine
            if (error.response?.status === 404) {
              loadEntries([])
              loadResults([])
              return
            }
            loadEntries([])
            loadResults([])
            return
          }
        }
      }

      if (!wheelId) {
        loadEntries([])
        loadResults([])
        return
      }

      try {
        // Load wheel with entries and results
        const wheelResponse = await wheelApi.get(wheelId)
        const wheel = wheelResponse.data

        // Load entries
        if (wheel.entries && Array.isArray(wheel.entries) && wheel.entries.length > 0) {
          const entries = wheel.entries.map((e: any) => ({
            id: e.id,
            text: e.text,
            imageUrl: e.imageUrl || null,
            weight: e.weight || null,
            color: e.color || null,
          }))
          loadEntries(entries)
        } else {
          loadEntries([])
        }

        // Load results
        if (wheel.results && Array.isArray(wheel.results)) {
          const results = wheel.results.map((r: any) => ({
            id: r.id,
            entryId: r.entryId,
            entryText: r.entry?.text || '',
            spunAt: new Date(r.spunAt),
          }))
          loadResults(results)
        } else {
          loadResults([])
        }

        // Load settings if available
        if (wheel.settings) {
          loadSettings(wheel.settings)
        }
      } catch (error) {
        // If wheel doesn't exist, clear the wheelId and localStorage
        setWheelId(null)
        localStorage.removeItem('spin-the-wheel_currentWheelId')
        loadEntries([])
        loadResults([])
      }
    }

    loadWheelData()
  }, [currentWheelId, loadEntries, loadResults, loadSettings, setWheelId])

  const handleRemoveWinner = () => {
    if (lastWinner) {
      if (settings.afterSpin.playClickSoundOnRemove) {
        try {
          const ctx = new (window.AudioContext || (window as any).webkitAudioContext)()
          const osc = ctx.createOscillator()
          const gain = ctx.createGain()
          osc.type = 'sine'
          osc.frequency.setValueAtTime(800, ctx.currentTime)
          gain.gain.setValueAtTime(0.1, ctx.currentTime)
          gain.gain.exponentialRampToValueAtTime(0.001, ctx.currentTime + 0.1)
          osc.connect(gain)
          gain.connect(ctx.destination)
          osc.start()
          osc.stop(ctx.currentTime + 0.1)
        } catch (e) {}
      }
      removeEntry(lastWinner.id)
      setLastWinner(null)
    }
  }

  return (
    <div className="app" style={{ background: settings.appearance.displayGradient ? 'linear-gradient(135deg, #e8f4f8 0%, #f5e8f5 100%)' : '#e8f4f8' }}>
      {/* Header */}
      <header className="app-header">
        <div className="header-logo">
          <div className="logo-icon">W</div>
          <h1>SPIN THE WHEEL</h1>
        </div>
        
        <div className="header-actions">
          <button 
            onClick={() => setShowCustomize(true)}
            className="btn-customize"
          >
            <Settings size={18} /> Customize
          </button>
        </div>
      </header>

      {/* Main Content */}
      <main className="app-main">
        {/* Wheel Area */}
        <div className="wheel-area">
          <WheelCanvas />
          
          {/* Winner Popup Overlay */}
          {lastWinner && settings.afterSpin.displayPopup && (
            <div className="winner-popup-overlay">
              <div className="winner-popup-card">
                <div className="winner-trophy-icon">
                  <Trophy size={48} />
                </div>
                <h2 className="winner-popup-title">
                  {settings.afterSpin.popupMessage}
                </h2>
                <div className="winner-name">
                  {lastWinner.text}
                </div>
                <div className="winner-popup-buttons">
                  <button 
                    onClick={() => setLastWinner(null)}
                    className="btn-popup-close"
                  >
                    Close
                  </button>
                  {settings.afterSpin.displayRemoveButton && (
                    <button 
                      onClick={handleRemoveWinner}
                      className="btn-popup-remove"
                    >
                      Remove
                    </button>
                  )}
                </div>
              </div>
            </div>
          )}
        </div>

        {/* Sidebar */}
        <aside className="sidebar">
          <EntryManager />
        </aside>
      </main>

      {/* Modals */}
      {showCustomize && (
        <CustomizationModal onClose={() => setShowCustomize(false)} />
      )}
    </div>
  )
}

export default App
