import { useEffect, useRef, useState, useCallback } from 'react'
import { useWheelStore } from '../../store/wheelStore'
import { WheelPhysics, getSecureRandomEntry, getSegmentIndexFromAngle } from '../../utils/wheelPhysics'
import { spinApi, wheelApi, resultApi, entryApi } from '../../services/api'
import confetti from 'canvas-confetti'
import './WheelCanvas.css'

export default function WheelCanvas() {
  const canvasRef = useRef<HTMLCanvasElement>(null)
  const physicsRef = useRef<WheelPhysics>(new WheelPhysics())
  const animationFrameRef = useRef<number>()
  
  // SINGLE SOURCE OF TRUTH: Store final angle and winner (calculated FROM angle, not predetermined)
  const frozenFinalAngleRef = useRef<number | null>(null)
  const frozenWinnerRef = useRef<{ entry: any; entryIndex: number } | null>(null)
  const currentWheelIdRef = useRef<string | null>(null) // Store wheelId for result saving

  const {
    entries,
    settings,
    isSpinning,
    setSpinning,
    setLastWinner,
    addResult,
    removeEntry,
    currentWheelId,
    setWheelId,
    loadEntries,
  } = useWheelStore()

  // Determine if we need backend for selection (large datasets)
  const needsBackend = entries.length > 1000
  const visibleEntries = entries.slice(0, settings.duringSpin.maxVisibleNames)
  const totalEntries = entries.length

  // Audio context for sounds
  const audioContextRef = useRef<AudioContext | null>(null)
  const lastTickIndexRef = useRef<number>(-1)

  const initAudio = useCallback(() => {
    if (!audioContextRef.current) {
      audioContextRef.current = new (window.AudioContext || (window as any).webkitAudioContext)()
    }
    if (audioContextRef.current.state === 'suspended') {
      audioContextRef.current.resume()
    }
    return audioContextRef.current
  }, [])

  const playTick = useCallback(() => {
    if (settings.duringSpin.sound === 'none' || settings.duringSpin.volume === 0) return
    try {
      const ctx = initAudio()
      const osc = ctx.createOscillator()
      const gain = ctx.createGain()
      
      // Snappy triangle tick
      osc.type = 'triangle'
      osc.frequency.setValueAtTime(600, ctx.currentTime)
      osc.frequency.exponentialRampToValueAtTime(150, ctx.currentTime + 0.04)
      
      gain.gain.setValueAtTime((settings.duringSpin.volume / 100) * 0.2, ctx.currentTime)
      gain.gain.exponentialRampToValueAtTime(0.001, ctx.currentTime + 0.04)
      
      osc.connect(gain)
      gain.connect(ctx.destination)
      
      osc.start()
      osc.stop(ctx.currentTime + 0.04)
    } catch (e) {}
  }, [settings.duringSpin.sound, settings.duringSpin.volume, initAudio])

  const playVictorySound = useCallback(() => {
    if (settings.afterSpin.sound === 'none' || settings.afterSpin.volume === 0) return
    try {
      const ctx = initAudio()
      // Cheerful arpeggio for victory
      const freqs = [392.00, 493.88, 587.33, 783.99] // G Major arpeggio
      freqs.forEach((freq, i) => {
        const osc = ctx.createOscillator()
        const gain = ctx.createGain()
        const startTime = ctx.currentTime + (i * 0.1)
        
        osc.type = 'sine'
        osc.frequency.setValueAtTime(freq, startTime)
        
        gain.gain.setValueAtTime(0, startTime)
        gain.gain.linearRampToValueAtTime((settings.afterSpin.volume / 100) * 0.1, startTime + 0.05)
        gain.gain.exponentialRampToValueAtTime(0.001, startTime + 0.8)
        
        osc.connect(gain)
        gain.connect(ctx.destination)
        
        osc.start(startTime)
        osc.stop(startTime + 0.8)
      })
    } catch (e) {}
  }, [settings.afterSpin.sound, settings.afterSpin.volume, initAudio])

  // Draw wheel function - PURE rendering, no winner logic
  const drawWheel = useCallback((angleRadians: number, spinning: boolean = false) => {
    const canvas = canvasRef.current
    if (!canvas) return

    const ctx = canvas.getContext('2d')
    if (!ctx) return

    // Get container size for responsive sizing
    const container = canvas.parentElement?.parentElement
    const containerWidth = container ? container.clientWidth : 600
    const containerHeight = container ? container.clientHeight : 600
    const containerSize = Math.min(containerWidth, containerHeight) - 64
    const size = Math.min(containerSize, 820)
    const dpr = window.devicePixelRatio || 1
    
    if (canvas.width !== size * dpr || canvas.height !== size * dpr) {
      canvas.width = size * dpr
      canvas.height = size * dpr
      canvas.style.width = `${size}px`
      canvas.style.height = `${size}px`
      ctx.scale(dpr, dpr)
    }

    const centerX = size / 2
    const centerY = size / 2
    const radius = size / 2 - 20

    ctx.clearRect(0, 0, size, size)

    // Handle tick sound timing based on rotation (only when spinning)
    if (spinning && visibleEntries.length > 0) {
      const PI2 = Math.PI * 2
      const anglePerEntry = PI2 / visibleEntries.length
      // Calculate which segment is at the pointer (0 radians / 3 o'clock)
      // Pointer is at 0, wheel starts at -PI/2, so: localAngle = PI/2 - angleRadians
      const localAngleAtPointer = ((Math.PI / 2) - angleRadians + PI2) % PI2
      const tickIndex = Math.floor(localAngleAtPointer / anglePerEntry)
      if (tickIndex !== lastTickIndexRef.current && lastTickIndexRef.current !== -1) {
        playTick()
      }
      lastTickIndexRef.current = tickIndex
    } else {
      lastTickIndexRef.current = -1
    }

    // Draw wheel shadow if enabled
    if (settings.appearance.wheelShadow) {
      ctx.save()
      ctx.shadowColor = 'rgba(0, 0, 0, 0.3)'
      ctx.shadowBlur = 20
      ctx.shadowOffsetX = 5
      ctx.shadowOffsetY = 5
      ctx.beginPath()
      ctx.arc(centerX, centerY, radius, 0, Math.PI * 2)
      ctx.fillStyle = '#fff'
      ctx.fill()
      ctx.restore()
    }

    // Save context for rotation
    ctx.save()
    ctx.translate(centerX, centerY)
    ctx.rotate(angleRadians) // Use radians directly

    // Draw wheel segments
    const PI2 = Math.PI * 2
    const anglePerEntry = PI2 / visibleEntries.length
    const colors = settings.appearance.colors

    visibleEntries.forEach((entry, index) => {
      // Start from -90° (top) and draw clockwise
      const startAngle = (index * anglePerEntry) - Math.PI / 2
      const endAngle = ((index + 1) * anglePerEntry) - Math.PI / 2

      // Draw segment
      ctx.beginPath()
      ctx.moveTo(0, 0)
      ctx.arc(0, 0, radius, startAngle, endAngle, false)
      ctx.closePath()

      // Use custom color if available, otherwise cycle through palette
      const color = entry.color || colors[index % colors.length]
      ctx.fillStyle = color
      ctx.fill()

      // Draw contour if enabled
      if (settings.appearance.contours) {
        ctx.strokeStyle = '#333'
        ctx.lineWidth = 2
        ctx.stroke()
      }

      // Draw text with clipping and dynamic font sizing
      ctx.save()
      
      // Create clipping path for this segment to prevent text overflow
      ctx.beginPath()
      ctx.moveTo(0, 0)
      ctx.arc(0, 0, radius, startAngle, endAngle, false)
      ctx.closePath()
      ctx.clip() // Clip all drawing to this segment boundary
      
      const textAngle = (startAngle + endAngle) / 2
      ctx.rotate(textAngle)
      
      // Calculate available arc length for text - use FULL width like wheelofnames
      const innerRadius = radius * 0.20 // Inner boundary (very close to center for maximum space)
      const outerRadius = radius * 0.98 // Outer boundary (very close to edge for maximum space)
      
      // Calculate the actual arc length at the outer radius (where text will extend to)
      // This gives us the maximum available width for text
      const arcLengthAtOuter = outerRadius * anglePerEntry
      const maxTextWidth = arcLengthAtOuter * 0.98 // Use 98% of full arc length
      
      // Find optimal font size to fit full text with letter spacing
      let fontSize = 24 // Start with base font size
      let optimalFontSize = fontSize
      const letterSpacing = 0.3 // Letter spacing in pixels (reduced for subtle effect)
      
      // Helper function to measure text width with letter spacing
      const measureTextWithSpacing = (text: string, font: string, spacing: number): number => {
        ctx.font = font
        const baseWidth = ctx.measureText(text).width
        // Add spacing between characters (not after last character)
        return baseWidth + (spacing * (text.length - 1))
      }
      
      // Measure text at base font size with letter spacing
      ctx.font = `${fontSize}px sans-serif`
      let textWidth = measureTextWithSpacing(entry.text, ctx.font, letterSpacing)
      
      // If text is too wide, reduce font size
      if (textWidth > maxTextWidth) {
        // Calculate optimal font size based on ratio
        optimalFontSize = Math.floor((fontSize * maxTextWidth) / textWidth)
        optimalFontSize = Math.max(12, optimalFontSize) // Minimum font size of 12px
        
        // Verify it fits with letter spacing
        ctx.font = `${optimalFontSize}px sans-serif`
        textWidth = measureTextWithSpacing(entry.text, ctx.font, letterSpacing)
        
        // Fine-tune: if still too wide, reduce more
        if (textWidth > maxTextWidth) {
          optimalFontSize = Math.floor((optimalFontSize * maxTextWidth) / textWidth)
          optimalFontSize = Math.max(12, optimalFontSize)
          ctx.font = `${optimalFontSize}px sans-serif`
          textWidth = measureTextWithSpacing(entry.text, ctx.font, letterSpacing)
        }
      } else {
        // Text fits, maximize font size to use full available width
        let testSize = fontSize + 1
        while (testSize <= 40) { // Max font size of 40px
          ctx.font = `${testSize}px sans-serif`
          const measuredWidth = measureTextWithSpacing(entry.text, ctx.font, letterSpacing)
          if (measuredWidth > maxTextWidth) {
            break
          }
          optimalFontSize = testSize
          testSize++
        }
      }
      
      // Set final text properties - no bold, with letter spacing, centered
      ctx.font = `${optimalFontSize}px sans-serif`
      ctx.textBaseline = 'middle'
      ctx.fillStyle = '#fff'
      ctx.shadowColor = 'rgba(0, 0, 0, 0.5)'
      ctx.shadowBlur = 3
      
      // Calculate center position for text (middle of the segment width)
      const textCenterRadius = (innerRadius + outerRadius) / 2
      
      // Draw text with letter spacing manually, centered
      // Calculate total width of text with spacing
      let totalTextWidth = 0
      for (let i = 0; i < entry.text.length; i++) {
        const charWidth = ctx.measureText(entry.text[i]).width
        totalTextWidth += charWidth
        if (i < entry.text.length - 1) {
          totalTextWidth += letterSpacing
        }
      }
      
      // Start position to center the text (use left align for individual characters)
      ctx.textAlign = 'left'
      let xPos = textCenterRadius - (totalTextWidth / 2)
      
      // Draw each character with proper spacing
      for (let i = 0; i < entry.text.length; i++) {
        ctx.fillText(entry.text[i], xPos, 0)
        // Move position by character width plus spacing
        const charWidth = ctx.measureText(entry.text[i]).width
        xPos += charWidth + letterSpacing
      }
      ctx.restore()
    })

    ctx.restore()

    // Draw center circle
    ctx.beginPath()
    ctx.arc(centerX, centerY, 30, 0, Math.PI * 2)
    ctx.fillStyle = '#fff'
    ctx.fill()
    ctx.strokeStyle = '#dadce0'
    ctx.lineWidth = 2
    ctx.stroke()


    // Draw pointer (fixed at right side, 0°)
    const pointerColor = settings.appearance.pointerChangesColor
      ? colors[0]
      : '#DB4437'
    ctx.fillStyle = pointerColor
    ctx.strokeStyle = '#fff'
    ctx.lineWidth = 2
    ctx.beginPath()
    ctx.moveTo(centerX + radius, centerY)
    ctx.lineTo(centerX + radius + 20, centerY - 15)
    ctx.lineTo(centerX + radius + 20, centerY + 15)
    ctx.closePath()
    ctx.fill()
    ctx.stroke()
  }, [visibleEntries, settings])

  // Auto-save wheel to backend
  const autoSaveWheel = useCallback(async () => {
    try {
      const wheelData = {
        name: 'My Wheel',
        settings,
        entries: entries.map((e, index) => ({
          text: e.text,
          imageUrl: e.imageUrl,
          weight: e.weight,
          color: e.color,
          order: index,
        })),
      }

      let wheelId = currentWheelId
      
      if (wheelId) {
        // Update existing wheel
        await wheelApi.update(wheelId, wheelData)
      } else {
        // Create new wheel
        const response = await wheelApi.create(wheelData)
        wheelId = response.data.id
        setWheelId(wheelId)
      }
      
      return wheelId
    } catch (error) {
      return currentWheelId
    }
  }, [entries, settings, currentWheelId, setWheelId, loadEntries])

  // Auto-save result to backend
  const autoSaveResult = useCallback(async (entry: any, wheelId: string | null) => {
    if (!wheelId) {
      return
    }
    
    try {
      // First, try to find the entry in the database by text (since frontend IDs may not match DB IDs)
      // If backend winner was used, it already has the correct DB entry ID
      let entryId = entry.id
      
      // If entry doesn't have a DB ID (frontend-only entry), we need to find it
      // For now, we'll use the entryApi to get entries and match by text
      if (!entryId || entryId.startsWith('temp-') || entryId.length < 10) {
        try {
          const entriesResponse = await entryApi.get(wheelId)
          const dbEntries = entriesResponse.data
          const matchingEntry = dbEntries.find((e: any) => e.text === entry.text)
          if (matchingEntry) {
            entryId = matchingEntry.id
          } else {
            return
          }
        } catch (err) {
          return
        }
      }
      
      await resultApi.add(wheelId, entryId)
    } catch (error: any) {
      // Silently fail - result saving is not critical
    }
  }, [])

  // Animation loop - ONLY handles animation, calculates winner FROM final angle
  useEffect(() => {
    if (!isSpinning) {
      // When not spinning, use frozen final angle if available
      const angleToDraw = frozenFinalAngleRef.current ?? physicsRef.current.getAngle()
      drawWheel(angleToDraw, false)
      return
    }

    const animate = (currentTime: number) => {
      // currentTime from requestAnimationFrame is in milliseconds (DOMHighResTimeStamp)
      const { angle, isComplete } = physicsRef.current.update(currentTime)
      drawWheel(angle, true)

      if (!isComplete) {
        animationFrameRef.current = requestAnimationFrame(animate)
      } else {
        // Spin complete - FREEZE the final angle
        const finalAngle = physicsRef.current.getAngle()
        frozenFinalAngleRef.current = finalAngle
        
        // Calculate winner FROM the final angle (like reference implementation)
        const winnerIndex = getSegmentIndexFromAngle(finalAngle, visibleEntries.length)
        const calculatedWinner = visibleEntries[winnerIndex]
        
        // Use backend winner if available (for large datasets), otherwise use visual winner
        const finalWinner = frozenWinnerRef.current?.entry || calculatedWinner
        
        if (finalWinner) {
          // FREEZE the winner (single source of truth)
          frozenWinnerRef.current = {
            entry: finalWinner,
            entryIndex: winnerIndex,
          }
          
          // Set winner in store (for popup display in App.tsx)
          setLastWinner(finalWinner)
          
          // Add result to local store
          addResult(finalWinner)
          
          // Auto-save result to backend (real-time sync)
          // Use ref first (from current spin), then fallback to store
          const wheelId = currentWheelIdRef.current || currentWheelId
          if (wheelId) {
            autoSaveResult(finalWinner, wheelId).catch(err => {
              // Silently handle result save error
            })
          } else {
            // Cannot save result: no wheelId available
          }

          // Play victory sound
          playVictorySound()

          // Launch confetti if enabled
          if (settings.afterSpin.launchConfetti) {
            confetti({
              particleCount: 180,
              spread: 80,
              origin: { y: 0.55 },
              colors: settings.appearance.colors,
            })
          }

          // Auto-remove winner if enabled
          if (settings.afterSpin.autoRemoveWinner) {
            setTimeout(() => {
              if (frozenWinnerRef.current) {
                removeEntry(frozenWinnerRef.current.entry.id)
                setLastWinner(null)
                frozenWinnerRef.current = null
              }
            }, settings.afterSpin.autoRemoveWinner * 1000)
          }
        }
        
        // Stop spinning
        setSpinning(false)
      }
    }

    // Start animation with current timestamp
    animationFrameRef.current = requestAnimationFrame(animate)

    return () => {
      if (animationFrameRef.current) {
        cancelAnimationFrame(animationFrameRef.current)
      }
    }
  }, [isSpinning, drawWheel, setSpinning, setLastWinner, addResult, settings, removeEntry, visibleEntries, playTick, playVictorySound, currentWheelId, autoSaveResult])

  // Continuous rendering when not spinning - keeps wheel at frozen position
  useEffect(() => {
    if (isSpinning) {
      return // Don't interfere with animation
    }

    // Use frozen final angle if available, otherwise current angle
    const angleToDraw = frozenFinalAngleRef.current ?? physicsRef.current.getAngle()
    
    const renderLoop = () => {
      // Always use the frozen final angle, never recalculate
      drawWheel(angleToDraw, false)
      animationFrameRef.current = requestAnimationFrame(renderLoop)
    }

    animationFrameRef.current = requestAnimationFrame(renderLoop)

    return () => {
      if (animationFrameRef.current) {
        cancelAnimationFrame(animationFrameRef.current)
      }
    }
  }, [isSpinning, drawWheel])

  // Handle window resize
  useEffect(() => {
    const handleResize = () => {
      if (!isSpinning) {
        const angleToDraw = frozenFinalAngleRef.current ?? physicsRef.current.getAngle()
        drawWheel(angleToDraw, false)
      }
    }

    window.addEventListener('resize', handleResize)
    return () => window.removeEventListener('resize', handleResize)
  }, [isSpinning, drawWheel])

  // Handle spin - Let physics run naturally, calculate winner FROM final angle
  const handleSpin = useCallback(async () => {
    if (isSpinning || entries.length === 0) return

    try {
      // Clear previous frozen data
      frozenFinalAngleRef.current = null
      frozenWinnerRef.current = null

      // Auto-save wheel before spinning (real-time sync)
      const wheelId = await autoSaveWheel()
      
      // Store wheelId in ref so it's available when result is saved
      currentWheelIdRef.current = wheelId
      
      // Update store if wheelId changed
      if (wheelId && wheelId !== currentWheelId) {
        setWheelId(wheelId)
        
        // Save wheelId to localStorage for persistence
        localStorage.setItem('spin-the-wheel_currentWheelId', wheelId)
        
        // Reload entries from database to get correct IDs
        try {
          const entriesResponse = await entryApi.get(wheelId)
          const dbEntries = entriesResponse.data.map((e: any) => ({
            id: e.id,
            text: e.text,
            imageUrl: e.imageUrl,
            weight: e.weight,
            color: e.color,
          }))
          loadEntries(dbEntries)
        } catch (error) {
          // Silently handle error
        }
      }

      // For large datasets (>1000), use backend for fair selection
      // For smaller datasets, use frontend physics
      let backendWinner: any = null
      if (needsBackend) {
        try {
          // Use wheelId if available, otherwise send entries array
          const spinPayload = wheelId 
            ? { wheelId }
            : { entries: entries.map(e => ({ id: e.id, text: e.text, weight: e.weight, color: e.color })) }
          
          const response = await spinApi.perform(spinPayload)
          backendWinner = response.data.winner
        } catch (error) {
          // Fallback to frontend if backend fails
        }
      }

      // Start natural physics spin (no predetermined target)
      // Physics will determine where it stops, then we calculate winner FROM that
      setSpinning(true)
      physicsRef.current.startSpin(settings.duringSpin.spinTime)
      
      // Store backend winner for later use if available
      if (backendWinner) {
        frozenWinnerRef.current = {
          entry: backendWinner,
          entryIndex: -1, // Will be determined by visual position
        }
      }
    } catch (error) {
      // Silently handle spin error
      setSpinning(false)
      frozenFinalAngleRef.current = null
      frozenWinnerRef.current = null
    }
  }, [
    isSpinning,
    entries,
    needsBackend,
    currentWheelId,
    settings,
    setSpinning,
    autoSaveWheel,
  ])

  // Keyboard shortcut
  useEffect(() => {
    const handleKeyPress = (e: KeyboardEvent) => {
      if ((e.ctrlKey || e.metaKey) && e.key === 'Enter' && !isSpinning) {
        handleSpin()
      }
    }

    window.addEventListener('keydown', handleKeyPress)
    return () => window.removeEventListener('keydown', handleKeyPress)
  }, [handleSpin, isSpinning])

  return (
    <div className="wheel-canvas-container">
      <div className="wheel-wrapper">
        <div className="wheel-inner">
          <canvas
            ref={canvasRef}
            className="wheel-canvas"
            onClick={handleSpin}
            style={{ cursor: isSpinning ? 'wait' : 'pointer' }}
          />
          {!isSpinning && entries.length > 0 && (
            <button 
              onClick={handleSpin}
              className="spin-button-overlay"
            >
              <div className="spin-button-inner">
                SPIN
              </div>
            </button>
          )}
        </div>
      </div>
      {needsBackend && !isSpinning && (
        <div className="wheel-instructions">
          <p className="backend-indicator">Using backend for {totalEntries} entries</p>
        </div>
      )}
    </div>
  )
}
