import { Request, Response } from 'express'
import prisma from '../services/prisma.js'
import { v4 as uuidv4 } from 'uuid'

export const createWheel = async (req: Request, res: Response) => {
  try {
    const { name, settings, entries } = req.body

    const wheel = await prisma.wheel.create({
      data: {
        name: name || 'Untitled Wheel',
        settings: settings || {},
        entries: {
          create: entries?.map((entry: any, index: number) => ({
            text: entry.text,
            imageUrl: entry.imageUrl,
            weight: entry.weight,
            color: entry.color,
            order: index,
          })) || [],
        },
      },
      include: {
        entries: true,
      },
    })

    res.json(wheel)
  } catch (error: any) {
    res.status(500).json({ error: error.message })
  }
}

export const getWheel = async (req: Request, res: Response) => {
  try {
    const { id } = req.params

    const wheel = await prisma.wheel.findUnique({
      where: { id },
      include: {
        entries: {
          orderBy: { order: 'asc' },
        },
        results: {
          include: {
            entry: true,
          },
          orderBy: { spunAt: 'desc' },
        },
      },
    })

    if (!wheel) {
      return res.status(404).json({ error: 'Wheel not found' })
    }

    res.json(wheel)
  } catch (error: any) {
    res.status(500).json({ error: error.message })
  }
}

export const updateWheel = async (req: Request, res: Response) => {
  try {
    const { id } = req.params
    const { name, settings, isPublic } = req.body

    const wheel = await prisma.wheel.update({
      where: { id },
      data: {
        ...(name && { name }),
        ...(settings && { settings }),
        ...(isPublic !== undefined && { isPublic }),
      },
    })

    res.json(wheel)
  } catch (error: any) {
    res.status(500).json({ error: error.message })
  }
}

export const deleteWheel = async (req: Request, res: Response) => {
  try {
    const { id } = req.params

    await prisma.wheel.delete({
      where: { id },
    })

    res.json({ message: 'Wheel deleted successfully' })
  } catch (error: any) {
    res.status(500).json({ error: error.message })
  }
}

export const getGalleryWheels = async (req: Request, res: Response) => {
  try {
    const wheels = await prisma.wheel.findMany({
      where: { isPublic: true },
      include: {
        entries: {
          take: 10, // Limit entries for gallery view
        },
      },
      orderBy: { createdAt: 'desc' },
      take: 20,
    })

    res.json(wheels)
  } catch (error: any) {
    res.status(500).json({ error: error.message })
  }
}

export const shareWheel = async (req: Request, res: Response) => {
  try {
    const { id } = req.params
    const shareToken = uuidv4()

    const wheel = await prisma.wheel.update({
      where: { id },
      data: { shareToken },
    })

    res.json({ shareToken, shareUrl: `/wheels/${shareToken}` })
  } catch (error: any) {
    res.status(500).json({ error: error.message })
  }
}

export const getRecentWheel = async (req: Request, res: Response) => {
  try {
    // Get the most recently updated wheel (or created if no updates)
    const wheel = await prisma.wheel.findFirst({
      orderBy: { updatedAt: 'desc' },
      include: {
        entries: {
          orderBy: { order: 'asc' },
        },
        results: {
          include: {
            entry: true,
          },
          orderBy: { spunAt: 'desc' },
        },
      },
    })

    if (!wheel) {
      return res.status(404).json({ error: 'No wheels found' })
    }

    res.json(wheel)
  } catch (error: any) {
    res.status(500).json({ error: error.message })
  }
}

