import crypto from 'crypto'

/**
 * Cryptographically secure random selection from an array
 * Uses crypto.randomBytes for true randomness
 */
export function getSecureRandomEntry<T>(entries: T[]): T {
  if (entries.length === 0) {
    throw new Error('Cannot select from empty array')
  }

  // Generate cryptographically secure random bytes
  const randomBytes = crypto.randomBytes(4)
  const randomValue = randomBytes.readUInt32BE(0) / (0xFFFFFFFF + 1)
  const index = Math.floor(randomValue * entries.length)

  return entries[index]
}

/**
 * Select random entry with weights (for advanced mode)
 */
export function getWeightedRandomEntry<T extends { weight?: number }>(
  entries: T[]
): T {
  if (entries.length === 0) {
    throw new Error('Cannot select from empty array')
  }

  // Calculate total weight
  const totalWeight = entries.reduce((sum, entry) => sum + (entry.weight || 1), 0)

  // Generate random value
  const randomBytes = crypto.randomBytes(4)
  const randomValue = (randomBytes.readUInt32BE(0) / (0xFFFFFFFF + 1)) * totalWeight

  // Find entry based on weighted random
  let currentWeight = 0
  for (const entry of entries) {
    currentWeight += entry.weight || 1
    if (randomValue <= currentWeight) {
      return entry
    }
  }

  // Fallback to last entry
  return entries[entries.length - 1]
}

