import { useRef } from 'react'
import { WheelSettings } from '../../types'
import { Play, Square } from 'lucide-react'
import './settings.css'

interface AfterSpinSettingsProps {
  settings: WheelSettings['afterSpin']
  onUpdate: (updates: Partial<WheelSettings['afterSpin']>) => void
}

export default function AfterSpinSettings({
  settings,
  onUpdate,
}: AfterSpinSettingsProps) {
  const audioContextRef = useRef<AudioContext | null>(null)
  const activeOscillatorsRef = useRef<OscillatorNode[]>([])

  const initAudio = () => {
    if (!audioContextRef.current) {
      audioContextRef.current = new (window.AudioContext || (window as any).webkitAudioContext)()
    }
    if (audioContextRef.current.state === 'suspended') {
      audioContextRef.current.resume()
    }
    return audioContextRef.current
  }

  const stopAllSounds = () => {
    activeOscillatorsRef.current.forEach(osc => {
      try {
        osc.stop()
        osc.disconnect()
      } catch (e) {}
    })
    activeOscillatorsRef.current = []
  }

  const playPreview = () => {
    if (settings.sound === 'none' || settings.volume === 0) return
    
    const ctx = initAudio()
    stopAllSounds()

    // Victory sounds (arpeggio)
    const freqs = [392.00, 493.88, 587.33, 783.99] // G Major arpeggio
    freqs.forEach((freq, i) => {
      const osc = ctx.createOscillator()
      const gain = ctx.createGain()
      const startTime = ctx.currentTime + (i * 0.12)
      
      osc.type = 'sine'
      osc.frequency.setValueAtTime(freq, startTime)
      
      gain.gain.setValueAtTime(0, startTime)
      gain.gain.linearRampToValueAtTime((settings.volume / 100) * 0.15, startTime + 0.05)
      gain.gain.exponentialRampToValueAtTime(0.001, startTime + 1.2)
      
      osc.connect(gain)
      gain.connect(ctx.destination)
      
      osc.start(startTime)
      osc.stop(startTime + 1.2)
      activeOscillatorsRef.current.push(osc)
    })
  }

  return (
    <div className="settings-panel">
      <div className="setting-group">
        <label className="setting-label">Sound</label>
        <div className="sound-control-group">
          <select
            value={settings.sound}
            onChange={(e) => onUpdate({ sound: e.target.value })}
            className="setting-select"
          >
            <option value="applause">Subdued applause</option>
            <option value="cheer">Cheer</option>
            <option value="none">None</option>
          </select>
          <div className="audio-preview-buttons">
            <button
              onClick={playPreview}
              className="audio-preview-btn"
              title="Play preview"
            >
              <Play size={16} fill="currentColor" />
            </button>
            <button
              onClick={stopAllSounds}
              className="audio-preview-btn"
              title="Stop"
            >
              <Square size={16} fill="currentColor" />
            </button>
          </div>
        </div>
      </div>

      <div className="setting-group">
        <label className="setting-label">Volume</label>
        <div className="slider-container">
          <input
            type="range"
            min="0"
            max="100"
            value={settings.volume}
            onChange={(e) => onUpdate({ volume: parseInt(e.target.value) })}
            className="setting-slider"
          />
          <span className="slider-value">{settings.volume}%</span>
        </div>
        <div className="slider-markers">
          <span>0%</span>
          <span>25%</span>
          <span>50%</span>
          <span>75%</span>
          <span>100%</span>
        </div>
      </div>

      <hr className="setting-separator" />

      <div className="setting-group">
        <label className="setting-checkbox">
          <input
            type="checkbox"
            checked={settings.animateWinningEntry}
            onChange={(e) =>
              onUpdate({ animateWinningEntry: e.target.checked })
            }
          />
          <span>Animate winning entry</span>
        </label>
        <label className="setting-checkbox">
          <input
            type="checkbox"
            checked={settings.launchConfetti}
            onChange={(e) => onUpdate({ launchConfetti: e.target.checked })}
          />
          <span>Launch confetti</span>
        </label>
        <label className="setting-checkbox">
          <input
            type="checkbox"
            checked={settings.autoRemoveWinner !== null}
            onChange={(e) =>
              onUpdate({ autoRemoveWinner: e.target.checked ? 5 : null })
            }
          />
          <span>Auto-remove winner after 5 seconds</span>
        </label>
      </div>

      <hr className="setting-separator" />

      <div className="setting-group">
        <label className="setting-checkbox">
          <input
            type="checkbox"
            checked={settings.displayPopup}
            onChange={(e) => onUpdate({ displayPopup: e.target.checked })}
          />
          <span>Display popup with message:</span>
        </label>
        {settings.displayPopup && (
          <input
            type="text"
            value={settings.popupMessage}
            onChange={(e) => onUpdate({ popupMessage: e.target.value })}
            className="setting-input"
            placeholder="We have a winner!"
          />
        )}
        <label className="setting-checkbox">
          <input
            type="checkbox"
            checked={settings.displayRemoveButton}
            onChange={(e) =>
              onUpdate({ displayRemoveButton: e.target.checked })
            }
          />
          <span>Display the "Remove" button</span>
        </label>
        <label className="setting-checkbox">
          <input
            type="checkbox"
            checked={settings.playClickSoundOnRemove}
            onChange={(e) =>
              onUpdate({ playClickSoundOnRemove: e.target.checked })
            }
          />
          <span>Play a click sound when the winner is removed</span>
        </label>
      </div>
    </div>
  )
}

