import { useState } from 'react'
import { useWheelStore } from '../../store/wheelStore'
import DuringSpinSettings from './DuringSpinSettings'
import AfterSpinSettings from './AfterSpinSettings'
import AppearanceSettings from './AppearanceSettings'
import './CustomizationModal.css'

interface CustomizationModalProps {
  onClose: () => void
}

export default function CustomizationModal({ onClose }: CustomizationModalProps) {
  const [activeTab, setActiveTab] = useState<'during' | 'after' | 'appearance'>('during')
  const { settings, updateSettings } = useWheelStore()

  const handleSave = () => {
    // Settings are already updated in the store via individual components
    onClose()
  }

  return (
    <div className="modal-overlay" onClick={onClose}>
      <div className="modal-content" onClick={(e) => e.stopPropagation()}>
        <div className="modal-header">
          <h2>Customize Wheel</h2>
          <button onClick={onClose} className="modal-close">
            ×
          </button>
        </div>

        <div className="modal-tabs">
          <button
            className={`modal-tab ${activeTab === 'during' ? 'active' : ''}`}
            onClick={() => setActiveTab('during')}
          >
            During spin
          </button>
          <button
            className={`modal-tab ${activeTab === 'after' ? 'active' : ''}`}
            onClick={() => setActiveTab('after')}
          >
            After spin
          </button>
          <button
            className={`modal-tab ${activeTab === 'appearance' ? 'active' : ''}`}
            onClick={() => setActiveTab('appearance')}
          >
            Appearance
          </button>
        </div>

        <div className="modal-body">
          {activeTab === 'during' && (
            <DuringSpinSettings
              settings={settings.duringSpin}
              onUpdate={(updates) =>
                updateSettings({ duringSpin: { ...settings.duringSpin, ...updates } })
              }
            />
          )}
          {activeTab === 'after' && (
            <AfterSpinSettings
              settings={settings.afterSpin}
              onUpdate={(updates) =>
                updateSettings({ afterSpin: { ...settings.afterSpin, ...updates } })
              }
            />
          )}
          {activeTab === 'appearance' && (
            <AppearanceSettings
              settings={settings.appearance}
              onUpdate={(updates) =>
                updateSettings({ appearance: { ...settings.appearance, ...updates } })
              }
            />
          )}
        </div>

        <div className="modal-footer">
          <button onClick={onClose} className="btn-cancel">
            Cancel
          </button>
          <button onClick={handleSave} className="btn-ok">
            OK
          </button>
        </div>
      </div>
    </div>
  )
}

