import { useRef, useEffect } from 'react'
import { WheelSettings } from '../../types'
import { Play, Square } from 'lucide-react'
import './settings.css'

interface DuringSpinSettingsProps {
  settings: WheelSettings['duringSpin']
  onUpdate: (updates: Partial<WheelSettings['duringSpin']>) => void
}

export default function DuringSpinSettings({
  settings,
  onUpdate,
}: DuringSpinSettingsProps) {
  // Clamp maxVisibleNames to valid range on mount
  useEffect(() => {
    if (settings.maxVisibleNames < 4 || settings.maxVisibleNames > 10000) {
      onUpdate({ maxVisibleNames: Math.max(4, Math.min(10000, settings.maxVisibleNames)) })
    }
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, []) // Only run on mount

  const audioContextRef = useRef<AudioContext | null>(null)
  const activeOscillatorsRef = useRef<OscillatorNode[]>([])

  // Logarithmic scale conversion for maxVisibleNames slider
  // Maps linear slider position (0-100) to logarithmic value (4-10000)
  const logToLinear = (value: number): number => {
    const min = 4
    const max = 10000
    const minLog = Math.log(min)
    const maxLog = Math.log(max)
    const valueLog = Math.log(value)
    return ((valueLog - minLog) / (maxLog - minLog)) * 100
  }

  const linearToLog = (linear: number): number => {
    const min = 4
    const max = 10000
    const minLog = Math.log(min)
    const maxLog = Math.log(max)
    const valueLog = minLog + (linear / 100) * (maxLog - minLog)
    return Math.round(Math.exp(valueLog))
  }

  const initAudio = () => {
    if (!audioContextRef.current) {
      audioContextRef.current = new (window.AudioContext || (window as any).webkitAudioContext)()
    }
    if (audioContextRef.current.state === 'suspended') {
      audioContextRef.current.resume()
    }
    return audioContextRef.current
  }

  const stopAllSounds = () => {
    activeOscillatorsRef.current.forEach(osc => {
      try {
        osc.stop()
        osc.disconnect()
      } catch (e) {}
    })
    activeOscillatorsRef.current = []
  }

  const playPreview = () => {
    if (settings.sound === 'none' || settings.volume === 0) return
    
    const ctx = initAudio()
    stopAllSounds()

    // Play a quick sequence of 3 ticks for preview
    for (let i = 0; i < 3; i++) {
      const startTime = ctx.currentTime + (i * 0.15)
      const osc = ctx.createOscillator()
      const gain = ctx.createGain()
      
      osc.type = 'triangle'
      osc.frequency.setValueAtTime(600, startTime)
      osc.frequency.exponentialRampToValueAtTime(150, startTime + 0.05)
      
      gain.gain.setValueAtTime((settings.volume / 100) * 0.3, startTime)
      gain.gain.exponentialRampToValueAtTime(0.001, startTime + 0.05)
      
      osc.connect(gain)
      gain.connect(ctx.destination)
      
      osc.start(startTime)
      osc.stop(startTime + 0.05)
      activeOscillatorsRef.current.push(osc)
    }
  }

  return (
    <div className="settings-panel">
      <div className="setting-group">
        <label className="setting-label">Sound</label>
        <div className="sound-control-group">
          <select
            value={settings.sound}
            onChange={(e) => onUpdate({ sound: e.target.value })}
            className="setting-select"
          >
            <option value="ticking">Ticking sound</option>
            <option value="none">None</option>
          </select>
          <div className="audio-preview-buttons">
            <button
              onClick={playPreview}
              className="audio-preview-btn"
              title="Play preview"
            >
              <Play size={16} fill="currentColor" />
            </button>
            <button
              onClick={stopAllSounds}
              className="audio-preview-btn"
              title="Stop"
            >
              <Square size={16} fill="currentColor" />
            </button>
          </div>
        </div>
      </div>

      <div className="setting-group">
        <label className="setting-label">Volume</label>
        <div className="slider-container">
          <input
            type="range"
            min="0"
            max="100"
            value={settings.volume}
            onChange={(e) => onUpdate({ volume: parseInt(e.target.value) })}
            className="setting-slider"
          />
          <span className="slider-value">{settings.volume}%</span>
        </div>
        <div className="slider-markers">
          <span>0%</span>
          <span>25%</span>
          <span>50%</span>
          <span>75%</span>
          <span>100%</span>
        </div>
      </div>

      <hr className="setting-separator" />

      <div className="setting-group">
        <label className="setting-checkbox">
          <input
            type="checkbox"
            checked={settings.displayDuplicates}
            onChange={(e) => onUpdate({ displayDuplicates: e.target.checked })}
          />
          <span>Display duplicates</span>
        </label>
        <label className="setting-checkbox">
          <input
            type="checkbox"
            checked={settings.spinSlowly}
            onChange={(e) => onUpdate({ spinSlowly: e.target.checked })}
          />
          <span>Spin slowly</span>
        </label>
        <label className="setting-checkbox">
          <input
            type="checkbox"
            checked={settings.showTitle}
            onChange={(e) => onUpdate({ showTitle: e.target.checked })}
          />
          <span>Show title</span>
        </label>
      </div>

      <hr className="setting-separator" />

      <div className="setting-group">
        <label className="setting-label">Spin time (seconds)</label>
        <div className="slider-container">
          <input
            type="range"
            min="1"
            max="60"
            value={settings.spinTime}
            onChange={(e) => onUpdate({ spinTime: parseInt(e.target.value) })}
            className="setting-slider"
          />
          <span className="slider-value">{settings.spinTime}</span>
        </div>
        <div className="slider-markers">
          <span>1</span>
          <span>10</span>
          <span>20</span>
          <span>30</span>
          <span>40</span>
          <span>50</span>
          <span>60</span>
        </div>
      </div>

      <hr className="setting-separator" />

      <div className="setting-group">
        <label className="setting-label">
          Max number of names visible on the wheel
        </label>
        <p className="setting-description">
          All names in the text-box have the same chance of winning, regardless
          of this value.
        </p>
        <div className="slider-container">
          <input
            type="range"
            min="0"
            max="100"
            step="0.1"
            value={logToLinear(Math.max(4, Math.min(10000, settings.maxVisibleNames)))}
            onChange={(e) => {
              const linearValue = parseFloat(e.target.value)
              const logValue = linearToLog(linearValue)
              onUpdate({ maxVisibleNames: Math.max(4, Math.min(10000, logValue)) })
            }}
            className="setting-slider"
            style={{
              background: `linear-gradient(to right, #2563eb 0%, #2563eb ${logToLinear(Math.max(4, Math.min(10000, settings.maxVisibleNames)))}%, #e2e8f0 ${logToLinear(Math.max(4, Math.min(10000, settings.maxVisibleNames)))}%, #e2e8f0 100%)`
            }}
          />
          <span className="slider-value">{Math.max(4, Math.min(10000, settings.maxVisibleNames))}</span>
        </div>
        <div className="slider-markers">
          <span>4</span>
          <span>100</span>
          <span>500</span>
          <span>1000</span>
          <span>5000</span>
          <span>10000</span>
        </div>
      </div>
    </div>
  )
}

