import { useState, useMemo, useRef } from 'react'
import { useWheelStore } from '../../store/wheelStore'
import { entryApi, wheelApi } from '../../services/api'
import { Plus, Shuffle, SortAsc, Trash2, X, Upload } from 'lucide-react'
import './EntryManager.css'

export default function EntryManager() {
  const [inputText, setInputText] = useState('')
  const [activeTab, setActiveTab] = useState<'entries' | 'results'>('entries')
  const [isUploading, setIsUploading] = useState(false)
  const [isSaving, setIsSaving] = useState(false)
  const fileInputRef = useRef<HTMLInputElement>(null)
  const {
    entries,
    results,
    settings,
    addEntry,
    bulkAddEntries,
    removeEntry,
    shuffleEntries,
    sortEntries,
    clearEntries,
    removeResult,
    clearResults,
    currentWheelId,
    setWheelId,
    loadEntries,
  } = useWheelStore()

  // Ensure wheel exists in database, create if needed
  const ensureWheel = async (): Promise<string | null> => {
    if (currentWheelId) {
      return currentWheelId
    }

    try {
      // Create new wheel with current entries and settings
      const wheelData = {
        name: 'My Wheel',
        settings,
        entries: entries.map((e, index) => ({
          text: e.text,
          imageUrl: e.imageUrl,
          weight: e.weight,
          color: e.color,
          order: index,
        })),
      }

      const response = await wheelApi.create(wheelData)
      const wheelId = response.data.id
      setWheelId(wheelId)
      
      // Save wheelId to localStorage for persistence
      localStorage.setItem('spin-the-wheel_currentWheelId', wheelId)
      
      // Reload entries from database to get correct IDs
      const entriesResponse = await entryApi.get(wheelId)
      const dbEntries = entriesResponse.data.map((e: any) => ({
        id: e.id,
        text: e.text,
        imageUrl: e.imageUrl,
        weight: e.weight,
        color: e.color,
      }))
      loadEntries(dbEntries)
      
      return wheelId
    } catch (error) {
      return null
    }
  }

  // Save entries to database
  const saveEntriesToDB = async (newEntries: string[]) => {
    try {
      const wheelId = await ensureWheel()
      if (!wheelId) {
        throw new Error('Failed to create or get wheel')
      }

      // Prepare entries for database
      const entriesToSave = newEntries.map((text, index) => ({
        text: text.trim(),
        order: entries.length + index,
      }))

      // Save to database
      await entryApi.add(wheelId, entriesToSave)
      
      // Reload entries from database to get correct IDs
      const entriesResponse = await entryApi.get(wheelId)
      const dbEntries = entriesResponse.data.map((e: any) => ({
        id: e.id,
        text: e.text,
        imageUrl: e.imageUrl,
        weight: e.weight,
        color: e.color,
      }))
      
      loadEntries(dbEntries)
    } catch (error: any) {
      throw error // Re-throw to handle in calling function
    }
  }

  const handleAdd = async () => {
    if (!inputText.trim() || isSaving) return

    setIsSaving(true)
    try {
      // Split by lines and add each entry
      const lines = inputText.split('\n').map(l => l.trim()).filter(l => l !== '')
      
      if (lines.length === 0) {
        setInputText('')
        setIsSaving(false)
        return
      }

      // Add to local store first (for immediate UI update)
      lines.forEach(line => addEntry(line))
      setInputText('')
      
      // Save to database
      await saveEntriesToDB(lines)
    } catch (error: any) {
      const errorMessage = error.response?.data?.error || error.message || 'Unknown error'
      alert(`Error saving entries: ${errorMessage}\n\nPlease check:\n1. Backend server is running\n2. Database connection is working`)
    } finally {
      setIsSaving(false)
    }
  }

  const parseCSV = (text: string): string[] => {
    const lines = text.split(/\r?\n/) // Handle both \n and \r\n
    const names: string[] = []
    let isFirstLine = true
    
    for (const line of lines) {
      const trimmed = line.trim()
      if (!trimmed) continue
      
      // Handle CSV format - split by comma and take first column (or whole line if no comma)
      // Also handle quoted values
      let name = ''
      if (trimmed.includes(',')) {
        // CSV format - try to parse properly
        const parts = trimmed.split(',').map(p => p.trim().replace(/^"|"$/g, ''))
        name = parts[0] || trimmed
      } else {
        // Plain text format
        name = trimmed
      }
      
      // Skip header row if it contains "Name" or "name"
      if (isFirstLine && (name.toLowerCase() === 'name' || name.toLowerCase() === 'names')) {
        isFirstLine = false
        continue
      }
      
      if (name) {
        names.push(name)
      }
      isFirstLine = false
    }
    
    return names
  }

  const handleFileUpload = async (event: React.ChangeEvent<HTMLInputElement>) => {
    const file = event.target.files?.[0]
    if (!file) return

    setIsUploading(true)
    
    try {
      const text = await file.text()
      const names = parseCSV(text)
      
      if (names.length === 0) {
        alert('No names found in the file. Please check the file format.')
        return
      }
      
      // Ensure wheel exists first
      const wheelId = await ensureWheel()
      if (!wheelId) {
        alert('Error: Could not create wheel. Please try again.')
        return
      }

      // Add all names in batches to avoid blocking the UI
      const batchSize = 500
      let imported = 0
      
      for (let i = 0; i < names.length; i += batchSize) {
        const batch = names.slice(i, i + batchSize)
        
        // Add to local store first
        bulkAddEntries(batch)
        imported += batch.length
        
        // Save batch to database
        try {
          const entriesToSave = batch.map((text, index) => ({
            text: text.trim(),
            order: entries.length + imported - batch.length + index,
          }))
          await entryApi.add(wheelId, entriesToSave)
        } catch (error) {
          // Silently handle batch save error
        }
        
        // Small delay to keep UI responsive for large files
        if (i + batchSize < names.length) {
          await new Promise(resolve => setTimeout(resolve, 50))
        }
      }
      
      // Reload all entries from database to get correct IDs
      try {
        const entriesResponse = await entryApi.get(wheelId)
        const dbEntries = entriesResponse.data.map((e: any) => ({
          id: e.id,
          text: e.text,
          imageUrl: e.imageUrl,
          weight: e.weight,
          color: e.color,
        }))
        loadEntries(dbEntries)
        } catch (error) {
          // Silently handle reload error
        }
      
      alert(`Successfully imported ${imported} names!`)
        } catch (error) {
          alert('Error reading file. Please make sure it is a valid CSV file.')
        } finally {
      setIsUploading(false)
      // Reset file input
      if (fileInputRef.current) {
        fileInputRef.current.value = ''
      }
    }
  }

  const handleUploadClick = () => {
    fileInputRef.current?.click()
  }

  // Handle remove entry - delete from database too
  const handleRemoveEntry = async (entryId: string) => {
    // Remove from local store first
    removeEntry(entryId)
    
    // Delete from database if wheelId exists
    if (currentWheelId) {
      try {
        await entryApi.remove(currentWheelId, entryId)
      } catch (error) {
        // Reload entries to sync state
        try {
          const entriesResponse = await entryApi.get(currentWheelId)
          const dbEntries = entriesResponse.data.map((e: any) => ({
            id: e.id,
            text: e.text,
            imageUrl: e.imageUrl,
            weight: e.weight,
            color: e.color,
          }))
          loadEntries(dbEntries)
        } catch (err) {
          // Silently handle reload error
        }
      }
    }
  }

  // Basic optimization: only render the first 500 entries for the list view
  const visibleEntries = useMemo(() => entries.slice(0, 500), [entries])

  return (
    <div className="entry-manager">
      <div className="entry-tabs">
        <button
          className={`entry-tab ${activeTab === 'entries' ? 'active' : ''}`}
          onClick={() => setActiveTab('entries')}
        >
          Entries ({entries.length})
        </button>
        <button
          className={`entry-tab ${activeTab === 'results' ? 'active' : ''}`}
          onClick={() => setActiveTab('results')}
        >
          Results ({results.length})
        </button>
      </div>

      <div className="entry-content">
        {activeTab === 'entries' ? (
          <>
            <div className="entry-input-section">
              <textarea
                value={inputText}
                onChange={(e) => setInputText(e.target.value)}
                placeholder="Enter names (one per line)..."
                className="entry-textarea-modern"
              />
              <div className="entry-buttons-row">
                <button
                  onClick={handleAdd}
                  className="btn-add-entries"
                  disabled={!inputText.trim() || isSaving}
                >
                  <Plus size={18} /> {isSaving ? 'Saving...' : 'Add Entries'}
                </button>
                <input
                  ref={fileInputRef}
                  type="file"
                  accept=".csv,.txt"
                  onChange={handleFileUpload}
                  style={{ display: 'none' }}
                />
                <button
                  onClick={handleUploadClick}
                  className="btn-upload-entries"
                  disabled={isUploading}
                >
                  <Upload size={18} /> {isUploading ? 'Uploading...' : 'Upload CSV'}
                </button>
              </div>
              
              <div className="entry-actions">
                <button 
                  onClick={shuffleEntries}
                  className="btn-action"
                >
                  <Shuffle size={14} /> Shuffle
                </button>
                <button 
                  onClick={sortEntries}
                  className="btn-action"
                >
                  <SortAsc size={14} /> Sort
                </button>
                <button 
                  onClick={async () => {
                    if (confirm('Are you sure you want to clear all entries?')) {
                      clearEntries()
                      // Clear from database if wheelId exists
                      if (currentWheelId) {
                        try {
                          await entryApi.clearAll(currentWheelId)
                        } catch (error) {
                          // Silently handle clear error
                        }
                      }
                    }
                  }}
                  className="btn-action btn-action-danger"
                >
                  <Trash2 size={14} /> Clear
                </button>
              </div>
            </div>
            
            <div className="entry-list-modern">
              {entries.length === 0 ? (
                <div className="empty-state-modern">
                  No entries yet.
                </div>
              ) : (
                <>
                  {visibleEntries.map((entry, idx) => (
                    <div key={entry.id} className="entry-item-modern">
                      <span className="entry-item-text">
                        <span className="entry-index">{idx + 1}.</span>
                        {entry.text}
                      </span>
                      <button 
                        onClick={() => handleRemoveEntry(entry.id)}
                        className="btn-remove-modern"
                      >
                        <X size={16} />
                      </button>
                    </div>
                  ))}
                  {entries.length > 500 && (
                    <div className="entry-more-indicator">
                      ... and {entries.length - 500} more entries
                    </div>
                  )}
                </>
              )}
            </div>
          </>
        ) : (
          <>
            <div className="results-header">
              <button 
                onClick={clearResults}
                className="btn-clear-results"
              >
                <Trash2 size={14} /> Clear History
              </button>
            </div>
            <div className="results-list-modern">
              {results.length === 0 ? (
                <div className="empty-state-modern">
                  No spin results yet.
                </div>
              ) : (
                results.map((result) => (
                  <div key={result.id} className="result-item-modern">
                    <div>
                      <div className="result-text-main">{result.entryText}</div>
                      <div className="result-timestamp">
                        {new Date(result.spunAt).toLocaleTimeString()}
                      </div>
                    </div>
                    <button 
                      onClick={() => removeResult(result.id)}
                      className="btn-remove-modern"
                    >
                      <X size={16} />
                    </button>
                  </div>
                ))
              )}
            </div>
          </>
        )}
      </div>
    </div>
  )
}

